﻿using System;
using System.IO;
using Ascon.Pilot.Bim.SDK.ModelTab.Toolbar;
using Ascon.Pilot.Bim.SDK.ModelTab;
using System.Linq;
using System.Windows.Forms;
using Ascon.Pilot.Bim.SDK.ModelViewer;
using Ascon.Pilot.Bim.SDK.CollisionJournal;
using System.Text;
using Ascon.Pilot.Bim.SDK.CollisionJournalSample.Tools;
using Ascon.Pilot.Bim.SDK.ModelStorage;
using Ascon.Pilot.SDK;
using System.Threading.Tasks;
using Ascon.Pilot.Bim.SDK.BottomPanelTabSample.Tools;
using System.Globalization;
using System.Collections.Generic;

namespace Ascon.Pilot.Bim.SDK.CollisionJournalSample
{
    public class ToolbarItemController : IDisposable
    {
        private IModelTab _modelTab;
        private readonly IObjectLoader _objectLoader;
        private readonly ICollisionJournalProvider _collisionJournalProvider;
        private readonly IModelStorageProvider _modelStorageProvider;
        private IModelStorage _modelStorage;
        private IToolbarCommandItem _toolbarCommand;
        private readonly byte[] _icon;
        private string _directory;

        private readonly IType _modelCheckerJournalsFolderType;
        private readonly IType _collisionJournalType;

        public ToolbarItemController(IModelTab modelTab, IObjectsRepository objectsRepository, ICollisionJournalProvider collisionJournalProvider, IModelStorageProvider modelStorageProvider)
        {
            _modelTab = modelTab;
            _objectLoader = new ObjectLoader(objectsRepository);
            _modelCheckerJournalsFolderType = objectsRepository.GetType(Constants.MODEL_CHECKER_JOURNALS_FOLDER_TYPE_NAME);
            _collisionJournalType = objectsRepository.GetType(Constants.COLLISION_JOURNAL_TYPE_NAME);
            _collisionJournalProvider = collisionJournalProvider;
            _modelStorageProvider = modelStorageProvider;
            _icon = IconLoader.GetIcon("toolbar_icon.svg");
        }

        public void AddToolbarItem()
        {
            var toolBarManager = _modelTab.GetToolbarManager();
            _toolbarCommand = AddToolbarButtonItem(toolBarManager);
        }

        public IModelViewer ModelViewer { get; private set; }

        public IToolbarButtonItem AddToolbarButtonItem(IToolbarManager toolBarManager)
        {
            toolBarManager.AddSeparator(toolBarManager.GetItems().Count());
            var buttonItem = toolBarManager.AddToolbarButtonItem(toolBarManager.GetItems().Count());
            buttonItem.Title = Properties.Resources.ExportJournalsToCsvString;
            buttonItem.IsEnabled = false;
            buttonItem.Icon = _icon;
            buttonItem.IsTitleVisible = false;
            buttonItem.Tooltip = Properties.Resources.ExportJournalsToCsvString;
            buttonItem.OnClick = ExportCollisionJournalsToCsv;

            return buttonItem;
        }

        private void ExportCollisionJournalsToCsv()
        {
            var result = ShowOpenDirectoryDialog(Properties.Resources.ChooseFolderString, out _directory);

            if (result != true)
                return;

            Task.Run(async () =>
            {
                var journalIds = await GetCollisionJournalIdsAsync();
                foreach (var journalId in journalIds)
                {
                    var collisionJournal = _collisionJournalProvider.GetJournal(journalId);
                    collisionJournal.Loaded += OnCollisionJournalLoaded;
                }
            });
        }

        private bool? ShowOpenDirectoryDialog(string description, out string directory)
        {
            var dialog = new FolderBrowserDialog();
            dialog.Description = description;
            directory = string.Empty;
            var result = dialog.ShowDialog();
            if (result == DialogResult.OK)
            {
                directory = dialog.SelectedPath;
                return true;
            }
            return false;
        }

        private async Task<IEnumerable<Guid>> GetCollisionJournalIdsAsync()
        {
            var collisionJournalIds = new List<Guid>();
            var queue = new Queue<Guid>();
            var modelObj = await _objectLoader.Load(ModelViewer.ModelId);

            FillQueue(queue, modelObj.TypesByChildren, collisionJournalIds);

            while (queue.Any())
            {
                var journalFolderId = queue.Dequeue();
                var journalFolder = await _objectLoader.Load(journalFolderId);
                FillQueue(queue, journalFolder.TypesByChildren, collisionJournalIds);
            }

            return collisionJournalIds;
        }

        private void FillQueue(Queue<Guid> queue, IDictionary<Guid, int> typesByChildren, List<Guid> result)
        {
            var journalIds = typesByChildren
                .Where(x => x.Value == _collisionJournalType.Id)
                .Select(x => x.Key);

            var journalFolderIds = typesByChildren
                .Where(x => x.Value == _modelCheckerJournalsFolderType.Id)
                .Select(x => x.Key);

            foreach (var journalFolderId in journalFolderIds)
            {
                queue.Enqueue(journalFolderId);
            }

            result.AddRange(journalIds);
        }

        private async void OnCollisionJournalLoaded(object sender, EventArgs e)
        {
            var collisionJournal = (ICollisionJournal)sender;
            var collisions = collisionJournal.LoadCollisions();
            var journalObject = await _objectLoader.Load(collisionJournal.JournalId);

            StringBuilder csvContent = new StringBuilder();
            csvContent.AppendLine($"{nameof(ICollision.Name)}," +
                                  $"{nameof(ICollision.FirstElementId)}," +
                                  $"{Constants.FIRST_ELEMENT_TAG_COLUMN_NAME}," +
                                  $"{nameof(ICollision.SecondElementId)}," +
                                  $"{Constants.SECOND_ELEMENT_TAG_COLUMN_NAME}," +
                                  $"{nameof(ICollision.FoundDateTime)}," +
                                  $"{nameof(ICollision.Volume)}");
            foreach (var collision in collisions)
            {
                var firstElementProperties = _modelStorage.LoadElementProperties(collision.FirstElementId, DateTime.Now);
                var secondElementProperties = _modelStorage.LoadElementProperties(collision.SecondElementId, DateTime.Now);

                var firstElementTag = firstElementProperties
                    .FirstOrDefault(x => x.Name == Constants.ELEMENT_INFORMATION_PROPERTIES_NAME)?.Properties
                    .FirstOrDefault(x => x.Name == Constants.TAG_PROPERTY_NAME)?.Value;
                var secondElementTag = secondElementProperties
                    .FirstOrDefault(x => x.Name == Constants.ELEMENT_INFORMATION_PROPERTIES_NAME)?.Properties
                    .FirstOrDefault(x => x.Name == Constants.TAG_PROPERTY_NAME)?.Value;
                var convertedVolume = collision.Volume + 1 < double.Epsilon
                    ? string.Empty
                    : collision.Volume.ToString(CultureInfo.InvariantCulture);

                csvContent.AppendLine($"{collision.Name}," +
                                      $"{collision.FirstElementId.ElementId}," +
                                      $"{firstElementTag}," +
                                      $"{collision.SecondElementId.ElementId}," +
                                      $"{secondElementTag}," +
                                      $"{collision.FoundDateTime}," +
                                      $"{convertedVolume}");
            }
            File.WriteAllText($"{_directory}/{journalObject.DisplayName}_{collisionJournal.JournalId}.csv", csvContent.ToString());
            UnloadCollisionJournal(collisionJournal);
        }

        private void UnloadCollisionJournal(ICollisionJournal collisionJournal)
        {
            collisionJournal.Loaded -= OnCollisionJournalLoaded;
            collisionJournal.Dispose();
        }

        public void OnModelLoaded(IModelViewer modelViewer)
        {
            if (_modelTab?.Id != modelViewer.TabId)
                return;

            ModelViewer = modelViewer;
            SubscribeModelStorage();
        }

        public void OnModelClosed(IModelViewer modelViewer)
        {
            if (_modelTab?.Id != modelViewer.TabId)
                return;

            ModelViewer = null;
            _toolbarCommand.IsEnabled = false;
        }

        private void SubscribeModelStorage()
        {
            _modelStorage = _modelStorageProvider.GetStorage(ModelViewer.ModelId);
            _modelStorage.Loaded += OnModelStorageLoaded;
        }

        private void OnModelStorageLoaded(object sender, EventArgs e)
        {
            _toolbarCommand.IsEnabled = true;
        }

        public void Dispose()
        {
            if (_modelStorage != null)
            {
                _modelStorage.Loaded -= OnModelStorageLoaded;
                _modelStorage.Dispose();
            }

            _toolbarCommand = null;
            _modelTab = null;
        }
    }
}
