/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using Ascon.Pilot.SDK.ObjectCard;
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Threading;

namespace Ascon.Pilot.SDK.ObjectCardHandle
{
    [Export(typeof(IObjectCardHandler)), 
     Export(typeof(IObjectCardValidationBeforeSubmitHandler)), 
     Export(typeof(IWorkflowObjectCardValidationBeforeSubmitHandler))]
    public class ObjectCardHandler : 
        IObjectCardHandler, 
        IObjectCardValidationBeforeSubmitHandler, 
        IWorkflowObjectCardValidationBeforeSubmitHandler
    {
        #region IObjectCardHandler implementation
        public bool Handle(IAttributeModifier modifier, ObjectCardContext context)
        {
            var isObjectModification = context.EditiedObject != null;
            if (isObjectModification || context.IsReadOnly)
                return false;

            var targetStrAttr = context.DisplayAttributes.FirstOrDefault(a => a.Type == AttributeType.String);
            if (targetStrAttr == null)
                return false;

            modifier.SetValue(targetStrAttr.Name, "plugin works");

            var targetAttr = context.DisplayAttributes.FirstOrDefault(a => a.Type == AttributeType.OrgUnit);
            if (targetAttr == null)
                return false;

            var valueToSet = new int[] { 5 };

            modifier.SetValue(targetAttr.Name, valueToSet);
            return true;
        }

        public bool OnValueChanged(IAttribute sender, AttributeValueChangedEventArgs args, IAttributeModifier modifier)
        {
            var currentAttributeValues = string.Empty;
            foreach (var displayAttribute in args.Context.DisplayAttributes)
            {
                if (args.Context.AttributesValues.ContainsKey(displayAttribute.Name))
                    currentAttributeValues += displayAttribute.Name == sender.Name
                        ? args.NewValue
                        : displayAttribute.Name + ": " + args.Context.AttributesValues[displayAttribute.Name] + Environment.NewLine;
            }

            if (args.Context.Type.Name == "Document" && sender.Name == "Sheet_number")
            {
                var newNameAttrValue = "Sheet no " + args.NewValue + "; " + (args.Context.EditiedObject == null ? " New object " : " Existed object");
                modifier.SetValue("Name", newNameAttrValue);
                return true;
            }

            return false;
        }
        #endregion

        #region ValidationBeforeSubmitHandler implementations
        public Task<bool> ValidateObjectCardBeforeSubmitAsync(ObjectCardContext context, CancellationToken token)
        {
            return Task<bool>.Factory.StartNew(() =>
            {
                var invalidValues = GetInvalidAttributesValues(context);
                if (invalidValues.Values.Any())
                {
                    ShowInvalidAttrsMessage(invalidValues);
                    return false;
                }

                return true;
            }, token);
        }

        public Task<bool> ValidateWorkflowObjectCardBeforeSubmitAsync(
            ObjectCardContext workflowCardContext,
            ObjectCardContext[][] stagesTaskCardContexts,
            CancellationToken token)
        {
            return Task<bool>.Factory.StartNew(() =>
            {
                var invalidValues = GetInvalidAttributesValues(workflowCardContext);

                var stage = 0;
                foreach (var stageContexts in stagesTaskCardContexts)
                {
                    stage++;

                    var task = 0;
                    foreach (var taskCardContext in stageContexts)
                    {
                        task++;

                        var taskInvalidValues = GetInvalidAttributesValues(taskCardContext);
                        foreach (var taskInvalidValue in taskInvalidValues)
                        {
                            invalidValues.Add($"Stage {stage}, task {task} {taskInvalidValue.Key}", taskInvalidValue.Value);
                        }
                    }
                }

                if (invalidValues.Values.Any())
                {
                    ShowInvalidAttrsMessage(invalidValues);
                    return false;
                }

                return true;
            }, token);
        }

        private Dictionary<string, string> GetInvalidAttributesValues(ObjectCardContext cardContext)
        {
            var invalidStingValue = "invalid";
            var invalidAttachmentId = Guid.Parse("91221f3b-4280-4aa7-bd25-6c9898933413");

            var result = new Dictionary<string, string>();
            foreach (var attr in cardContext.AttributesValues)
            {
                if(attr.Value as string == invalidStingValue)
                    result.Add(attr.Key, attr.Value as string);

                if (attr.Key == SystemTaskAttributes.ATTACHMENTS)
                {
                    if(attr.Value is Guid[] ids && ids.Any(x => x == invalidAttachmentId))
                        result.Add(attr.Key, invalidAttachmentId.ToString());
                }
            }

            return result;
        }

        private void ShowInvalidAttrsMessage(Dictionary<string, string> invalidAttrs)
        {
            var valuesAsDisplayString = invalidAttrs.Select(x => $"{x.Key} - {x.Value}");

            Application.Current.Dispatcher.BeginInvoke(new Action(() =>
            {
                Application.Current.Dispatcher.BeginInvoke(new Action(() =>
                {
                    MessageBox.Show(
                        caption: "Extension validation result",
                        messageBoxText: $"Invalid attribute values:{Environment.NewLine}{string.Join(Environment.NewLine, valuesAsDisplayString)}");
                }), DispatcherPriority.Background);
            }), DispatcherPriority.ApplicationIdle);
            
        }
        #endregion
    }
}
