/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.ComponentModel.Composition;
using System.Linq;
using System.Windows;
using System.Windows.Media;
using Ascon.Pilot.SDK.Menu;
using Ascon.Pilot.SDK.ObjectsSample.ObjectCardView;
using Ascon.Pilot.Theme.ColorScheme;

namespace Ascon.Pilot.SDK.ObjectsSample
{
    [Export(typeof(INewTabPage)), Export(typeof(IMenu<ObjectsViewContext>))]
    public class ObjectsSamplePlugin : INewTabPage, IMenu<ObjectsViewContext>
    {
        private readonly IObjectsRepository _repository;
        private readonly ITabServiceProvider _tabServiceProvider;
        private readonly IPilotDialogService _pilotDialogService;
        private readonly IObjectModifier _modifier;
        private readonly IConnectionStateProvider _connectionStateProvider;
        private readonly IAttributeFormatParser _attributeFormatParser;

        [ImportingConstructor]
        public ObjectsSamplePlugin(
            IObjectsRepository repository, 
            ITabServiceProvider tabServiceProvider, 
            IPilotDialogService pilotDialogService, 
            IAttributeFormatParser attributeFormatParser,
            IObjectModifier modifier,
            IConnectionStateProvider connectionStateProvider)
        {
            _repository = repository;
            _tabServiceProvider = tabServiceProvider;
            _pilotDialogService = pilotDialogService;
            _modifier = modifier;
            _connectionStateProvider = connectionStateProvider;
            _attributeFormatParser = attributeFormatParser;

            var convertFromString = ColorConverter.ConvertFromString(pilotDialogService.AccentColor);
            if (convertFromString != null)
            {
                var accentColor = (Color)convertFromString;
                ColorScheme.Initialize(accentColor, pilotDialogService.Theme);
            }
        }

        public void BuildNewTabPage(INewTabPageHost host)
        {
            host.AddButton("Objects plugin sample", "OpenExtensionCommand", "Objects plugin sample", null);
        }

        public void OnButtonClick(string name)
        {
            if (name != "OpenExtensionCommand") 
                return;
            
            var title = _tabServiceProvider.GetActiveTabPageTitle();
            _tabServiceProvider.UpdateTabPageContent(title, "Objects plugin sample", GetMainView());
        }

        private FrameworkElement GetMainView()
        {
            var view = new MainView();
            var viewModel = new MainViewModel(_repository, _pilotDialogService, _tabServiceProvider, _attributeFormatParser, _modifier, _connectionStateProvider);
            view.DataContext = viewModel;
            return view;
        }

        private const string CREATE_BASED_ON_ITEM_ID = "miCreateBasedOn";

        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            var selection = context.SelectedObjects.ToList();
            var selectedSourceObject = !context.IsContext && selection.Count == 1 ? selection[0] : null;
            if(selectedSourceObject == null)
                return;

            builder.AddItem(CREATE_BASED_ON_ITEM_ID, index: 0)
                .WithHeader("Create new based on this item");
        }

        public void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            if (CREATE_BASED_ON_ITEM_ID != name)
                return;

            var selection = context.SelectedObjects.ToList();
            var selectedSourceObject = !context.IsContext && selection.Count == 1 ? selection[0] : null;
            if (selectedSourceObject == null)
                return;

            var type = _repository.GetType(selectedSourceObject.Type.Id);
            _modifier.Clear();
            var builder = _modifier.Create(selectedSourceObject.ParentId, type);
            foreach (var attr in selectedSourceObject.Type.Attributes)
            {
                if(attr.IsService)
                    continue;

                var sourceAttrValueObj = selectedSourceObject.Attributes.TryGetValue(attr.Name, out var value)
                    ? value
                    : null;

                if(sourceAttrValueObj == null)
                    continue;

                builder.SetAttributeAsObject(attr.Name, sourceAttrValueObj);

                var isReferenceBookWithRelationAttr =
                    _attributeFormatParser.TryParseReferenceBookConfiguration(attr.Configuration2(),
                        out var attributeConfiguration) && attributeConfiguration.CreateLink;

                var referenceBookRelation = isReferenceBookWithRelationAttr
                    ? selectedSourceObject.Relations.FirstOrDefault(x => x.Name == attr.Name)
                    : null;

                if (referenceBookRelation != null)
                {
                    builder.AddRelation(
                        referenceBookRelation.Name, 
                        referenceBookRelation.Type, 
                        Guid.Empty, 
                        referenceBookRelation.TargetId,
                        referenceBookRelation.VersionId);
                }
            }

            var showDocumentPreview = selectedSourceObject.Type.HasFiles;

            _pilotDialogService.ShowObjectDialog(
                parentId: selectedSourceObject.ParentId,
                objectTypeId: selectedSourceObject.Type.Id, 
                modifier: _modifier,
                showDocumentPreview: showDocumentPreview);
            _modifier.Clear();
        }
    }
}
