/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Threading.Tasks;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Automation;

namespace ChangeChildrenStateActivity
{
    [Export(typeof(IAutomationActivity))]
    public class ChangeChildrenStateActivity : AutomationActivity
    {
        private string _attributeName;
        private readonly Dictionary<Guid, IUserStateMachine> _stateMachines;
        private readonly Dictionary<Guid, IUserState> _userStates;
        private string AttributeName => _attributeName ?? (_attributeName = this.GetValue<string>(AutomationDefaults.AttributeName));

        [ImportingConstructor]
        public ChangeChildrenStateActivity(IObjectsRepository repository)
        {
            _stateMachines = repository.GetUserStateMachines().ToDictionary(k => k.Id, v => v);
            _userStates = repository.GetUserStates().ToDictionary(k => k.Id, v => v);
        }

        public override string Name => nameof(ChangeChildrenStateActivity);
        
        public override Task RunAsync(IObjectModifier modifier, IAutomationBackend backend, IAutomationEventContext context, TriggerType triggerType)
        {
            var source = context.Source;

            if (!source.Attributes.TryGetValue(AttributeName, out var stateTo))
                return Task.CompletedTask;

            var attr = context.Source.Type.Attributes.FirstOrDefault(x => x.Name == AttributeName);
            if (attr == null)
                return Task.CompletedTask;

            var userStateTo = _userStates[(Guid)stateTo];

            foreach (var childId in source.Children)
            {
                var child = backend.GetObject(childId);
                var transitions = GetAvailableTransitions(attr, child.Attributes, context.InitiatingPerson);
                if (transitions.Any(x => x.StateTo == userStateTo.Id))
                {
                    modifier.EditById(childId).SetAttribute(AttributeName, (Guid)stateTo);
                }

            }

            return Task.CompletedTask;
        }

        private IEnumerable<ITransition> GetAvailableTransitions(IAttribute attribute, IDictionary<string, object> attributes, IPerson person)
        {
            var currentState = GetStateValue(attributes, attribute.Name);
            var allTransitions = GetAllTransitions(currentState, attribute);
            return allTransitions.Where(t => IsTransitionAvailable(t, attributes, person)).ToList();
        }

        private Guid GetStateValue(IDictionary<string, object> attributes, string attributeName)
        {
            if (attributes == null)
                return SystemStates.TASK_NONE_STATE_ID;

            return attributes.TryGetValue(attributeName, out var value)
                ? value as Guid? ?? SystemStates.TASK_NONE_STATE_ID
                : SystemStates.TASK_NONE_STATE_ID;
        }

        private IEnumerable<ITransition> GetAllTransitions(Guid stateFrom, IAttribute attribute)
        {
            if (!Guid.TryParse(attribute.Configuration2(), out var stateMachineId))
                return Enumerable.Empty<ITransition>();

            var stateMachine = _stateMachines[stateMachineId];

            if (!stateMachine.StateTransitions.Any())
                return Enumerable.Empty<ITransition>();

            if (!stateMachine.StateTransitions.TryGetValue(stateFrom, out var transitions))
                return Enumerable.Empty<ITransition>();

            return transitions ?? Enumerable.Empty<ITransition>();
        }

        private bool IsTransitionAvailable(ITransition transition, IDictionary<string, object> attributes, IPerson person)
        {
            if (transition.AvailableForPositionsSource.NullOrEmpty())
                return true;

            var permissionsInfo = new AttributePermissionInfo(transition.AvailableForPositionsSource);
            return permissionsInfo.ExtractAllOrgUnits(attributes).Intersect(person.AllOrgUnits()).Any();
        }
    }

    static class StringExtensions
    {
        public static bool NullOrEmpty(this string[] arr)
        {
            return arr == null || !arr.Any();
        }
    }
}
