/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Threading;
using System.Windows.Input;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Extensions;
using ShowObjectsTree.Async.Annotations;
using ShowObjectsTree.Async.Tools;

namespace ShowObjectsTree.Async
{
    class ObjectsTreeViewModel : INotifyPropertyChanged
    {
        private readonly IObjectsRepository _repository;
        private readonly CancellationTokenSource _cts = new CancellationTokenSource();
        
        public ObjectsTreeViewModel(IObjectsRepository repository)
        {
            _repository = repository;
            Nodes = new ObservableCollection<Node>();
            CancelCommand = new RelayCommand(o => !_cts.IsCancellationRequested, o => _cts.Cancel());
        }

        public ObservableCollection<Node> Nodes { get; }
        public ICommand CancelCommand { get; }

        public void Load(IEnumerable<Guid> ids)
        {
            LoadCore(ids, Nodes);
        }

        private async void LoadCore(IEnumerable<Guid> ids, ObservableCollection<Node> destination)
        {
            try
            {
                var objs = await _repository.GetObjectsAsync(ids, _cts.Token);
                foreach (var node in objs.Select(CreateNode))
                {
                    destination.Add(node);
                    LoadCore(node.Children, node.Subnodes);
                }
            }
            catch (OperationCanceledException)
            {
                // Cancellation was requested
            }
        }

        private Node CreateNode(IDataObject dataObject)
        {
            return new Node(dataObject.Id, dataObject.DisplayName, dataObject.Type.SvgIcon, dataObject.Children);
        }

        #region INotifyPropertyChanged

        public event PropertyChangedEventHandler PropertyChanged;

        [NotifyPropertyChangedInvocator]
        protected virtual void OnPropertyChanged([CallerMemberName] string propertyName = null)
        {
            PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(propertyName));
        }

        #endregion
    }
}
