﻿/*
  Copyright © 2025 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/

/*Пример добавляет новое действие DocumentVersionsStateManager, с помощью которого версии документа будет назначено соответсвующее состояние.
При подписании запроса на подпись с ролью "В производство работ" актуальная версия будет переведена в состояние "В производство работ", а предыдущая
версия - в состояние "Аннулирован". При создании новой версии предыдущая версия будет переведена в состояние "Аннулирован".

Исходные данные:
В Pilot-myAdmin в конфигурации базы данных у типа "document" необходимо указать группу состояний для версий документов, 
в которой нужно добавить состояния intoProduction и annulled с возможностью перехода из none в intoProduction, 
из none в annulled и из intoProduction в annulled.

Данный скрипт будет запускать действие DocumentVersionsStateManager при наступлении одного из следующих событий:
1. DocumentSigned - документ был подписан
2. VersionChanged - была создана новая версия документа*/

using System;
using System.Linq;
using System.Threading.Tasks;
using System.ComponentModel.Composition;
using Ascon.Pilot.Common;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.DataModifier;
using Ascon.Pilot.ServerExtensions.SDK;

namespace DocumentVersionsStateManager
{
    [Export(typeof(IServerActivity))]
    public class DocumentVersionsStateManager : IServerActivity
    {
        public string Name => "DocumentVersionsStateManager";
        public const string IntoProductionStateName = "intoProductionStateName";
        public const string AnnulledStateName = "annulledStateName";
        public const string IntoProductionRole = "intoProductionRole";

        public Task RunAsync(IModifierBase modifier, IModifierBackend backend, IServerActivityContext serverActivityContext, IAutomationEventContext automationEventContext)
        {
            var source = automationEventContext.Source;
            if (!serverActivityContext.Params.TryGetValue(IntoProductionStateName, out var intoProductionStateName))
                throw new InvalidOperationException($"{IntoProductionStateName} parameter is not defined in DocumentVersionsStateManager activity");
            if (!serverActivityContext.Params.TryGetValue(AnnulledStateName, out var annulledStateName))
                throw new InvalidOperationException($"{AnnulledStateName} parameter is not defined in DocumentVersionsStateManager activity");
            if (!serverActivityContext.Params.TryGetValue(IntoProductionRole, out var intoProductionRoleValue))
                throw new InvalidOperationException($"{IntoProductionRole} parameter is not defined in DocumentVersionsStateManager activity");

            var metaData = backend.GetMetadata();

            var intoProduction = metaData.UserStates.FirstOrDefault(x => x.Name == (string)intoProductionStateName);
            if (intoProduction == null)
                throw new InvalidOperationException($"State {IntoProductionStateName} not found");

            var annulledState = metaData.UserStates.FirstOrDefault(x => x.Name == (string)annulledStateName);
            if (annulledState == null)
                throw new InvalidOperationException($"State {AnnulledStateName} not found");

            var actualFile = GetActualDocumentFile(source, out var snapShotCreated);
            var isFullySigned = IsFullySigned(actualFile);

            if (isFullySigned && IsSignedByIntoProductionRole(actualFile, (string)intoProductionRoleValue))
            {
                modifier.EditObject(source).SetFileState(actualFile.Id, intoProduction.Id, snapShotCreated);
            }

            var previousFile = GetPreviousDocumentFile(source, out snapShotCreated);
            if (previousFile != null && previousFile.StateId != annulledState.Id)
                modifier.EditObject(source).SetFileState(previousFile.Id, annulledState.Id, snapShotCreated);

            return Task.CompletedTask;
        }

        public static bool IsFullySigned(INFile actualFile)
        {
            if (actualFile == null || actualFile.SignatureRequests.Count == 0)
                return false;

            var requestCount = actualFile.SignatureRequests.Count;
            var signedRequest = actualFile.SignatureRequests.Count(req => req.Signs.Any() && !req.IsAdvancementRequired);
            return signedRequest == requestCount;
        }

        public static bool IsSignedByIntoProductionRole(INFile file, string intoProductionRoleValue)
        {
            if (file == null || file.SignatureRequests.Count == 0)
                return false;

            return file
                .SignatureRequests
                .Any(req =>
                                req.Signs.Any()
                             && req.Role != null 
                             && req.Role.Equals(intoProductionRoleValue, StringComparison.OrdinalIgnoreCase));
        }

        private static INFile GetActualDocumentFile(INObject @object, out DateTime snapshotCreated)
        {
            var actualSnapShot = @object.ActualFileSnapshot;
            snapshotCreated = actualSnapShot.Created;
            return actualSnapShot.Files.FirstOrDefault(f => FileExtensionHelper.IsXpsOrPdfAlike(f.Name));
        }

        private static INFile GetPreviousDocumentFile(INObject @object, out DateTime snapshotCreated)
        {
            var previousSnapshot = @object.PreviousFileSnapshots.OrderByDescending(s => s.Created).FirstOrDefault();
            snapshotCreated = previousSnapshot?.Created ?? DateTime.MinValue;

            return previousSnapshot?.Files.FirstOrDefault(f => FileExtensionHelper.IsXpsOrPdfAlike(f.Name));
        }
    }
}