/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Packaging;
using System.Windows.Annotations;
using System.Windows.Annotations.Storage;

namespace Pilot.Xps.Domain.Annotations
{
    class AnnotationsStore : IDisposable
    {
        private readonly XmlStreamStore _instance;

        public AnnotationsStore(Package package)
        {
            _instance = GetStreamStoreAnnotations(package);
        }

        public void AddAnnotation(Annotation annotation)
        {
            _instance.AddAnnotation(annotation);
        }

        public void DeleteAnnotation(Annotation annotation)
        {
            _instance.DeleteAnnotation(annotation.Id);
        }

        public void DeleteAnnotations()
        {
            foreach (var annotation in GetAnnotations())
            {
                DeleteAnnotation(annotation);
            }
        }

        public Annotation GetAnnotation(Annotation annotation)
        {
            return GetAnnotation(annotation.Id);
        }

        public Annotation GetAnnotation(Guid annotationId)
        {
            return _instance.GetAnnotation(annotationId);
        }

        public IEnumerable<Annotation> GetAnnotations()
        {
            return _instance.GetAnnotations();
        }

        public static XmlStreamStore GetStreamStoreAnnotations(Package package)
        {
            PackagePart annotationPart = GetAnnotationPart(package);
            Stream annotationStream = annotationPart.GetStream();
            return new XmlStreamStore(annotationStream);
        }

        private static PackagePart GetAnnotationPart(Package xpsPackage)
        {
            PackagePart packagePart = xpsPackage.GetPart(GetFixedDocumentSequenceUri(xpsPackage));
            PackageRelationship annotationRelationship = null;
            foreach (PackageRelationship rel in packagePart.GetRelationshipsByType(AnnotationConstants.ANNOT_RELS_TYPE))
            {
                annotationRelationship = rel;
            }

            PackagePart annotationPart;

            if (annotationRelationship == null)
            {
                xpsPackage.DeletePart(new Uri(AnnotationConstants.ANNOTATION_FILE, UriKind.Relative));
                annotationPart = xpsPackage.CreatePart(PackUriHelper.CreatePartUri(new Uri(AnnotationConstants.ANNOTATION_FILE, UriKind.Relative)), AnnotationConstants.ANNOT_CONTENT_TYPE);
                packagePart.CreateRelationship(annotationPart.Uri, TargetMode.Internal, AnnotationConstants.ANNOT_RELS_TYPE);
            }
            else
            {
                annotationPart = xpsPackage.GetPart(annotationRelationship.TargetUri);
            }

            return annotationPart;
        }

        private static Uri GetFixedDocumentSequenceUri(Package package)
        {
            foreach (PackagePart packagePart in package.GetParts())
            {
                if (packagePart.ContentType == AnnotationConstants.FIXED_DOCUMENT_SEQUENCE_CONTENT_TYPE)
                    return packagePart.Uri;
            }

            return null;
        }

        public void Dispose()
        {
            _instance.Flush();
            _instance.Dispose();
        }
    }
}
