/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System.IO;
using System.Windows;
using System.Windows.Markup;
using System.Windows.Media.Imaging;
using System.Xml;
using System.Xml.Linq;

namespace Pilot.Xps.Domain.GraphicLayer
{
    class GraphicElementContentFactory : IGraphicElementContentFactory
    {
        public IGraphicLayerElementContent MakeContent(string contentType, Stream stream)
        {
            switch (contentType)
            {
                case XpsConstants.BITMAP:
                    var bitmap = CreateBitmapFromStream(stream);
                    return new RasterGraphicLayerElementContent(bitmap, stream);
                case XpsConstants.XAML:
                    using (var reader = new StreamReader(stream))
                    {
                        var control = reader.ReadToEnd();
                        var uiControl = ParseToControl(control);
                        return new XamlGraphicLayerElementContent(uiControl);
                    }
                default:
                    return null;
            }
        }

        private static FrameworkElement ParseToControl(string uiElement)
        {
            if (string.IsNullOrEmpty(uiElement))
                return null;
            try
            {
                FrameworkElement tmp = null;
                var el = XElement.Parse(uiElement);
                uiElement = el.ToString();
                var nsmgr = new XmlNamespaceManager(new NameTable());
                nsmgr.AddNamespace(string.Empty, @"http://schemas.microsoft.com/winfx/2006/xaml/presentation");
                var xmlParserContext = new XmlParserContext(null, nsmgr, null, XmlSpace.None);
                var reader = XmlReader.Create(new StringReader(uiElement), null, xmlParserContext);
                tmp = XamlReader.Load(reader) as FrameworkElement;
                return tmp;
            }
            catch
            {
                return null;
            }
        }

        private BitmapImage CreateBitmapFromStream(Stream stream)
        {
            var bitmap = new BitmapImage();
            bitmap.BeginInit();
            bitmap.StreamSource = stream;
            bitmap.CacheOption = BitmapCacheOption.OnLoad;
            bitmap.EndInit();
            bitmap.Freeze();
            return bitmap;
        }
    }

    interface IGraphicElementContentFactory
    {
        IGraphicLayerElementContent MakeContent(string contentType, Stream stream);
    }
}
