/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using Ascon.Pilot.Theme.Controls;
using Microsoft.Practices.Prism.Commands;

namespace PilotShare.Client.ToolbarItems
{
    public static class ToolbarBuilderExtensions
    {
        public static IToolbarBuilder AddSeparator(this IToolbarBuilder builder)
        {
            builder.AddSeparator(builder.Count);
            return builder;
        }

        public static IToolbarButtonItemBuilder AddButtonItem(this IToolbarBuilder builder, string name)
        {
            return builder.AddButtonItem(name, builder.Count);
        }
    }

    public class SubmenuRequestedEventArgs : EventArgs
    {
        public IToolbarBuilder Builder { get; private set; }

        public SubmenuRequestedEventArgs(IToolbarBuilder builder)
        {
            Builder = builder;
        }
    }

    public interface IToolbarBuilder
    {
        void AddSeparator(int index);
        IToolbarButtonItemBuilder AddButtonItem(string name, int index);

        int Count { get; }
        IEnumerable<string> ItemNames { get; }
        void Clear();
        void RemoveItemAt(int index);
        IToolbarButtonItemBuilder GetButtonItem(string name);
        //IToolbarMenuButtonItemBuilder GetMenuItem(string name);
        //IToolbarToggleButtonItemBuilder GetToggleItem(string name);
        //IToolbarInputItemBuilder GetInputItem(string name);
    }


    public class ToolbarBuilder : IToolbarBuilder
    {
        private readonly IList _items;

        public ToolbarBuilder(IList items)
        {
            _items = items;
        }

        public void AddSeparator(int index)
        {
            if (_items.Count <= 0)
                return;

            if (!IsCorrectIndex(index))
                return;

            _items.Insert(index, new ToolbarSeparator());
        }

        public IToolbarButtonItemBuilder AddButtonItem(string name, int index)
        {
            var item = new ToolbarButtonItem(name);
            if (IsCorrectIndex(index))
                _items.Insert(index, item);

            return item;
        }

        public int Count => _items.Count;

        public IEnumerable<string> ItemNames
        {
            get
            {
                var items = _items.Cast<IToolbarItem>();
                return items.Select(i => i.Name);
            }
        }

        public IToolbarButtonItemBuilder GetButtonItem(string name)
        {
            var item = _items.OfType<ToolbarButtonItem>().FirstOrDefault(i => i.Name == name);
            return item == null ? null : GetToolbarItemBuilder<IToolbarButtonItemBuilder>(item);
        }

        //public IToolbarMenuButtonItemBuilder GetMenuItem(string name)
        //{
        //    var item = _items.OfType<ToolbarMenuButtonItem>().FirstOrDefault(i => i.Name == name);
        //    return item == null ? null : GetToolbarItemBuilder<IToolbarMenuButtonItemBuilder>(item);
        //}

        public void Clear()
        {
            _items.Clear();
        }

        public void RemoveItemAt(int index)
        {
            if (IsCorrectIndex(index))
                _items.RemoveAt(index);
        }

        private static T GetToolbarItemBuilder<T>(ToolbarButtonItem item) where T : class
        {
            var command = item.Command as DelegateCommand;
            if (command == null)
                return item as T;

            if (!string.IsNullOrEmpty(command.Caption))
                item.WithHeader(command.Caption);

            if (command.Icon != null)
                item.WithIcon(command.Icon);

            return item as T;
        }

        private static bool IsCorrectIndex(int index)
        {
            return index >= 0;
        }
    }
}
