/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using Ascon.Pilot.Common.DataProtection;
using Ascon.Pilot.Server.Api;
using Ascon.Pilot.Server.Api.Contracts;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Hosting;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Hosting;
using Microsoft.Extensions.Logging;
using PilotShare.Server.Middleware;
using PilotShare.Server.Models;

namespace PilotShare.Server
{
    public class Startup
    {
        public Startup(IConfiguration configuration)
        {
            Configuration = configuration;
        }

        public IConfiguration Configuration { get; }

        // This method gets called by the runtime. Use this method to add services to the container.
        public void ConfigureServices(IServiceCollection services)
        {
            var authSection = Configuration.GetSection("AuthSettings");
            services.Configure<AuthSettings>(authSection);
            services.AddSingleton<IServerVersionInfoService, ServerVersionInfoService>();

            var authSettings = authSection.Get<AuthSettings>();
            var serverUrl = authSettings.Url;
            var login = authSettings.Login;
            var password = authSettings.Password;
            var credentials = ConnectionCredentials.GetConnectionCredentials(serverUrl, login, password.ConvertToSecureString());
            var remoteServices = new RemoteServices(credentials, authSettings.LicenseCode);
            var serverApi = remoteServices.GetServerApi();
            services.AddSingleton<IServerApi>(serverApi);

            var linkStateProvider = new LinkStateProvider(serverApi);
            services.AddScoped<ILinkStateProvider>((s) => linkStateProvider);

            var fileArchiveApi = remoteServices.GetFileArchiveApi();
            var fileLoader = new FileLoader(fileArchiveApi);
            var filesLoadService = new FileLoaderService(linkStateProvider, fileLoader);
            services.AddScoped<IFileLoaderService>(s => filesLoadService);

            services.AddControllersWithViews();
        }

        // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
        public void Configure(IApplicationBuilder app, IWebHostEnvironment env, ILoggerFactory loggerFactory)
        {
            if (env.IsDevelopment())
            {
                app.UseDeveloperExceptionPage();
            }
            else
            {
                app.UseExceptionHandler("/error");
                // The default HSTS value is 30 days. You may want to change this for production scenarios, see https://aka.ms/aspnetcore-hsts.
                app.UseHsts();
            }

            loggerFactory.AddLog4Net();
            app.UseStatusCodePages(async context =>
            {
                //if (context.HttpContext.Response.StatusCode == 400)
                //{
                //    context.HttpContext.Response.Redirect("~/Views/Shared/Errors/AccessDenied.cshtml");
                //}
                if (context.HttpContext.Response.StatusCode == 404)
                {
                    context.HttpContext.Response.Redirect("/error404");
                }
            });
            app.UseExceptionHandlerMiddleware();
            
            app.UseHttpsRedirection();
            app.UseStaticFiles();

            app.UseRouting();

            app.UseEndpoints(endpoints =>
            {
                endpoints.MapControllerRoute(
                    name: "default",
                    pattern: "{controller=Home}/{action=Index}/{id?}");
            });
        }
    }
}
