﻿/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;

namespace Ascon.Pilot.Bim.SDK.ModelSamplesHelpers
{
    public static class Parsers
    {
        public static Guid? GetModelPartId(string id)
        {
            if (Guid.TryParse(id, out var modelPartId))
            {
                return modelPartId;
            }

            Messages.ShowErrorMessage("Model part id invalid");
            return null;
        }

        public static IModelElementId GetModelElement(string id, string modelPartId)
        {
            if (Guid.TryParse(id, out var elementId))
            {
                if (Guid.TryParse(modelPartId, out var elementModelPartId))
                {
                    return new ModelElementId(elementId, elementModelPartId);
                }

                Messages.ShowErrorMessage("Model part id invalid");
            }
            else
                Messages.ShowErrorMessage("Element id invalid");

            return null;
        }

        public static IMeshDefinitionId GetMeshDefinition(string id, string modelPartId)
        {
            if (Guid.TryParse(id, out var meshId))
            {
                if (Guid.TryParse(modelPartId, out var elementModelPartId))
                {
                    return new MeshDefinitionId(meshId, elementModelPartId);
                }

                Messages.ShowErrorMessage("Model part id invalid");
            }
            else
                Messages.ShowErrorMessage("Mesh id invalid");

            return null;
        }


        public static List<IModelElementId> GetModelElements(string ids)
        {
            var elements = new List<IModelElementId>();
            var regex = new Regex("{([^:]+):([^:]+)}");
            var matches = regex.Matches(ids);

            foreach (Match match in matches)
            {
                var id = match.Groups[1];
                var modelPartId = match.Groups[2];

                var element = GetModelElement(id.Value, modelPartId.Value);

                if (element != null)
                    elements.Add(element);
            }

            return elements;
        }

        public static List<IMeshDefinitionId> GetMeshDefinitions(string ids)
        {
            var elements = new List<IMeshDefinitionId>();
            var regex = new Regex("{([^:]+):([^:]+)}");
            var matches = regex.Matches(ids);

            foreach (Match match in matches)
            {
                var id = match.Groups[1];
                var modelPartId = match.Groups[2];

                var element = GetMeshDefinition(id.Value, modelPartId.Value);

                if (element != null)
                    elements.Add(element);
            }

            return elements;
        }
    }
}
