/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Data;

namespace Ascon.Pilot.SDK.Controls.ObjectCardView.States
{
    static class StateComboBoxBehavior
    {
        public static readonly DependencyProperty ViewModelProperty =
            DependencyProperty.RegisterAttached("ViewModel", typeof(CardControlViewModel), typeof(StateComboBoxBehavior), new PropertyMetadata(null, OnViewModelChanged));

        public static CardControlViewModel GetViewModel(DependencyObject obj)
        {
            return (CardControlViewModel) obj.GetValue(ViewModelProperty);
        }

        public static void SetViewModel(DependencyObject obj, CardControlViewModel value)
        {
            obj.SetValue(ViewModelProperty, value);
        }

        private static void OnViewModelChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var comboBox = d as ComboBox;
            if (comboBox == null)
                throw new NotSupportedException();

            var viewModel = (CardControlViewModel) e.NewValue;

            comboBox.Loaded += (o, a) =>
            {
                comboBox.ItemsSource = GetStatuses(viewModel);
                comboBox.IsEditable = false;
                comboBox.DisplayMemberPath = nameof(StateItem.Title);
                comboBox.SelectedValuePath = nameof(StateItem.Name);
                comboBox.SetBinding(Selector.SelectedValueProperty, new Binding(nameof(CardControlViewModel.Value)));
                comboBox.IsTextSearchCaseSensitive = false;

                if (comboBox.HasItems && comboBox.SelectedItem == null)
                    comboBox.SelectedItem = comboBox.Items[0];
            };
        }

        private static IEnumerable GetStatuses(CardControlViewModel viewModel)
        {
            var repository = viewModel.Repository;
            var availableStates = new List<Guid>();
            var currentState = viewModel.Value as Guid? ?? SystemStates.TASK_NONE_STATE_ID;
            if (currentState != SystemStates.TASK_NONE_STATE_ID || !viewModel.Attribute.IsObligatory)
                availableStates.Add(currentState);

            var currentPerson = repository.GetCurrentPerson();
            var manager = viewModel.TransitionManager;
            var transitions = manager.GetAvailableTransitions(viewModel.Attribute, GetInitiatorPosition(viewModel.Type, currentPerson), currentPerson);
            foreach (var transition in transitions)
            {
                availableStates.Add(transition.StateTo);
            }

            return availableStates.Select(x =>
            {
                var state = repository.GetUserStates().First(y => y.Id == x);
                return new StateItem(state);
            }).ToList();
        }

        private static Dictionary<string, object> GetInitiatorPosition(IType type, IPerson person)
        {
            var result = new Dictionary<string, object>();
            if (type.Attributes.Any(a => a.Name == SystemTaskAttributes.INITIATOR_POSITION))
            {
                var value = person.MainPosition.Position;
                result[SystemTaskAttributes.INITIATOR_POSITION] = new[] {value};
            }

            return result;
        }
    }
}
