﻿/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading;
using System.Xml;
using System.Xml.Serialization;

namespace Ascon.Pilot.SDK.CryptoProviderSample
{
    [Export(typeof(ICryptoProvider))]
    [Export(typeof(IAutomationSigner))]
    public class CryptoProvider : ICryptoProvider, IAutomationSigner
    {
        private const string SIG_EXTENSION = "sig";
        private const string SIG_ALGORITHM = "MD5";
        private readonly IObjectModifier _modifier;
        private readonly IFileProvider _fileProvider;
        private readonly IObjectsRepository _repository;


        [ImportingConstructor]
        public CryptoProvider(IObjectModifier modifier, IFileProvider fileProvider, IObjectsRepository repository)
        {
            _modifier = modifier;
            _fileProvider = fileProvider;
            _repository = repository;
        }

        public void ReadSignature(IDataObject document, Stream fileStream, IFilesSnapshot snapshot,
            ISignatureRequest signatureRequest,
            IReadSignatureListener listener)
        {
            if (!signatureRequest.Signs.Any() || !Guid.TryParse(signatureRequest.Signs.Last(), out var signId))
                return;

            var signFile = snapshot.Files.FirstOrDefault(x => x.Id == signId);
            if (signFile == null)
                return;

            try
            {
                var orgUnit = _repository.GetOrganisationUnit(signatureRequest.PositionId);
                var signature = GetSignature(signFile);

                var signerName = string.IsNullOrEmpty(orgUnit.Title) ? signature.Signer : $"{signature.Signer} ({orgUnit.Title})";
                listener.SetSignerName(signerName);
                listener.SetSignDate(signature.SignDate.ToLocalTime().ToString("g"));

                var documentData = StreamToArray(fileStream);
                var md5 = CreateMD5(documentData);

                if (md5.SequenceEqual(signature.MD5))
                {
                    listener.SignatureIsValid();
                    listener.CertificateIsValid();
                    return;
                }

                listener.SignatureIsInvalid("Incorrect hash");
            }
            catch (Exception ex)
            {
                listener.SignatureIsInvalid(ex.Message);
            }
            finally
            {
                listener.ReadCompleted();
            }
        }

        public void ReadImportedSignature(Stream file, byte[] sigBase64, IImportedSignatureListener listener)
        {
            try
            {
                var signature = GetSignature(sigBase64);
                var documentData = StreamToArray(file);
                var md5 = CreateMD5(documentData);

                if (md5.SequenceEqual(signature.MD5))
                {
                    listener.SignatureIsValid();
                    listener.CertificateIsValid();
                    listener.SetPublicKeyOid(SIG_ALGORITHM);
                    listener.SetCadesType(CadesType.NotCades);
                    return;
                }

                listener.SignatureIsInvalid("Incorrect hash");
            }
            catch (Exception ex)
            {
                listener.SignatureIsInvalid(ex.Message);
            }
            finally
            {
                listener.ReadCompleted();
            }
        }

        public IEnumerable<ICertificate> GetCertificates()
        {
            var list = new List<ICertificate>();
            foreach (var file in Directory.EnumerateFiles(
                         Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "CertStorage")))
            {
                var cert = DeserializeCrt(file);
                cert.PublicKeyOid = SIG_ALGORITHM;
                list.Add(cert);
            }

            return list;
        }
        public void Sign(IObjectModifier modifier, Guid documentId, IFile file, Stream stream, ICertificate cert, IEnumerable<ISignatureRequest> signatureRequests)
        {
            SignCore(modifier, documentId, file, stream, cert, signatureRequests);
        }
        public void Sign(Guid documentId, IFile file, Stream stream, ICertificate cert,
            IEnumerable<ISignatureRequest> signatureRequests)
        {
            _modifier.Clear();
            SignCore(_modifier, documentId, file, stream, cert, signatureRequests);

            _modifier.Apply();
        }

        private void SignCore(IObjectModifier modifier, Guid documentId, IFile file, Stream stream, ICertificate cert, IEnumerable<ISignatureRequest> signatureRequests)
        {
            var builder = modifier.EditById(documentId);

            var documentData = StreamToArray(stream);
            var md5 = CreateMD5(documentData);

            var signature = new Signature()
            {
                MD5 = md5,
                SignDate = DateTime.UtcNow,
                Signer = cert.Subject
            };

            var serializedSign = Serialize(signature);
            var signBse64 = Convert.ToBase64String(serializedSign);

            foreach (var request in signatureRequests)
            {
                using (var ms = new MemoryStream(Encoding.UTF8.GetBytes(signBse64)))
                {
                    builder.AddFile($"{file.Name}{request.Id}.{SIG_EXTENSION}", ms, DateTime.UtcNow,
                        DateTime.UtcNow, DateTime.UtcNow, out var signatureFileId);

                    builder.SetSignatures(f => f.Id == file.Id).SetPublicKeyOid(request.Id, cert.PublicKeyOid)
                        .SetLastSignCadesType(request.Id, CadesType.NotCades)
                        .SetSign(request.Id, signatureFileId.ToString());
                }
            }
        }

        public bool IsAlgorithmSupported(string algorithmOid)
        {
            return algorithmOid == SIG_ALGORITHM;
        }

        public bool IsAlgorithmSupported(byte[] sigBase64)
        {
            try
            {
                var sigData = ConvertFormBase64(sigBase64);
                using (var ms = new MemoryStream(sigData))
                {
                    {
                        var sign = Deserialize<Signature>(ms);
                        return sign != null;
                    }
                }
            }
            catch (Exception)
            {
                return false;
            }
        }
        private static byte[] CreateMD5(byte[] input)
        {
            using (System.Security.Cryptography.MD5 md5 = System.Security.Cryptography.MD5.Create())
            {
                byte[] hashBytes = md5.ComputeHash(input);
                return hashBytes;
            }
        }

        private Signature GetSignature(IFile file)
        {
            using (var signatureStream = _fileProvider.OpenRead(file))
            {
                var sigBase64 = StreamToArray(signatureStream);
                var data = ConvertFormBase64(sigBase64);

                using (var ms = new MemoryStream(data))
                {
                    return Deserialize<Signature>(ms);
                }
            }
        }

        private Signature GetSignature(byte[] sigBase64)
        {
            var data = ConvertFormBase64(sigBase64);
            using (var ms = new MemoryStream(data))
            {
                return Deserialize<Signature>(ms);
            }
        }

        private byte[] ConvertFormBase64(byte[] data)
        {
            string base64String = Encoding.UTF8.GetString(data);
            return Convert.FromBase64String(base64String);
        }

        private byte[] StreamToArray(Stream stream)
        {
            using (var ms = new MemoryStream())
            {
                stream.CopyTo(ms);
                return ms.ToArray();
            }
        }

        private byte[] Serialize<T>(T obj)
        {
            XmlSerializer serializer = new XmlSerializer(typeof(T));
            using (var sww = new StringWriter())
            {
                using (XmlTextWriter writer = new XmlTextWriter(sww) { Formatting = Formatting.Indented })
                {
                    serializer.Serialize(writer, obj);
                    return System.Text.Encoding.Unicode.GetBytes(sww.ToString()); 
                }
            }
        }

        private FakeCertificate DeserializeCrt(string path)
        {
            using (FileStream fs = File.Open(path, FileMode.Open))
            {
                return Deserialize<FakeCertificate>(fs);
            }
        }

        private T Deserialize<T>(Stream stream)
        {
            XmlSerializer serializer = new XmlSerializer(typeof(T));
            return (T)serializer.Deserialize(stream);
        }
    }
}
