/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using System.Xml.Serialization;
using Ascon.Pilot.SDK.Controls;
using Ascon.Pilot.SDK.GraphicLayerSample;
using Ascon.Pilot.SDK.Menu;

namespace Ascon.Pilot.SDK.QRCodeSample
{
    [Export(typeof(IMenu<ObjectsViewContext>))]
    public class QrCodeSample : IMenu<ObjectsViewContext>, IObserver<INotification>
    {
        private static string MENU_ITEM_NAME = "QR_CODE_MENU_ITEM_NAME";
        private static string QR_CODE_ELEMENT_NAME = "QrCode";

        private readonly IObjectModifier _modifier;
        private readonly IObjectsRepository _repository;
        private readonly IFileProvider _fileProvider;
        private readonly IPerson _currentPerson;

        [ImportingConstructor]
        public QrCodeSample(IObjectModifier modifier, IObjectsRepository repository, IFileProvider fileProvider)
        {
            _modifier = modifier;
            _repository = repository;
            _fileProvider = fileProvider;
            _currentPerson = repository.GetCurrentPerson();

            repository.SubscribeNotification(NotificationKind.ObjectCreated).Subscribe(this);
            repository.SubscribeNotification(NotificationKind.ObjectFileChanged).Subscribe(this);
        }

        public async void OnNext(INotification notification)
        {
            var loader = new ObjectLoader(_repository);
            var obj = await loader.Load(notification.ObjectId);

            if (notification.UserId != _currentPerson.Id || obj.Files.Any(f => f.Name.Contains(Constants.QR_CODE_FILE_TAG)))
                return;

            CreateOrReplaceQrCode(obj);
        }

        private void CreateOrReplaceQrCode(IDataObject obj)
        {
            var configuration = TryParseXml(obj.Type.Configuration());
            var qrCodeElement = configuration?.Elements().LastOrDefault(e =>
                e.Name.LocalName.Equals(QR_CODE_ELEMENT_NAME, StringComparison.InvariantCultureIgnoreCase));

            if (qrCodeElement == null)
                return;

            var qrCode = new QrCode(qrCodeElement);
            var qrConfigFile = obj.Files.FirstOrDefault(f => f.Name.Contains(Constants.QR_CODE_FILE_TAG));
            if (qrConfigFile == null)
                AddNewQrCode(qrCode, obj.Id, obj.ActualFileSnapshot.Created.Ticks);
            else
                EditExistsQrCode(qrCode, obj, qrConfigFile);
        }

        private void EditExistsQrCode(QrCode qrCode, IDataObject obj, IFile existQrFile)
        {
            var builder = _modifier.EditById(obj.Id);
            var existQrCode = GetExistsQrFile(existQrFile);
            var existBitmapFile = GetExistsBitmapFile(obj, existQrCode.GetContentFileName());

            try
            {
                using (var s = qrCode.GetBitmapStream(obj.Id, obj.ActualFileSnapshot.Created.Ticks))
                    builder.AddOrReplaceFile(existBitmapFile.Name, s, existBitmapFile, DateTime.Now, DateTime.Now, DateTime.Now);

                using (var stream = new MemoryStream())
                {
                    var element = qrCode.GetGraphicLayerElement(existQrCode.ElementId, existQrCode.ContentId);
                    new XmlSerializer(typeof(GraphicLayerElement)).Serialize(stream, element);
                    builder.AddOrReplaceFile(existQrFile.Name, stream, existQrFile, DateTime.Now, DateTime.Now, DateTime.Now);
                }
                _modifier.Apply();
            }
            catch (Exception)
            {
                _modifier.Clear();
                throw;
            }
        }

        private IFile GetExistsBitmapFile(IDataObject obj, string bitmap)
        {
            var existBitmapFile = obj.Files.FirstOrDefault(f => f.Name.Contains(bitmap));
            if (existBitmapFile == null)
                throw new Exception("Bitmap for qrCode isn't exist!");
            return existBitmapFile;
        }

        private GraphicLayerElement GetExistsQrFile(IFile existQrFile)
        {
            var serializer = new XmlSerializer(typeof(GraphicLayerElement));
            using (var oldFileStream = _fileProvider.OpenRead(existQrFile))
            {
                var oldQrCode = serializer.Deserialize(oldFileStream) as GraphicLayerElement;
                if (oldQrCode == null)
                    throw new Exception("QrCode isn't exist!");

                return oldQrCode;
            }
        }

        private void AddNewQrCode(QrCode qrCode, Guid objId, long version)
        {
            var builder = _modifier.EditById(objId);

            var element = qrCode.GetGraphicLayerElement(Guid.NewGuid(), Guid.NewGuid());
            using (var stream = new MemoryStream())
            {
                var qrName = $"{element.GetFileName()}_{Constants.QR_CODE_FILE_TAG}";
                new XmlSerializer(typeof(GraphicLayerElement)).Serialize(stream, element);
                builder.AddFile(qrName, stream, DateTime.Now, DateTime.Now, DateTime.Now);
            }

            using (var s = qrCode.GetBitmapStream(objId, version))
                builder.AddFile(element.GetContentFileName(), s, DateTime.Now, DateTime.Now, DateTime.Now);
            _modifier.Apply();
        }

        private XElement TryParseXml(string xml)
        {
            try
            {
                return XElement.Parse(xml);
            }
            catch (Exception)
            {
                return null;
            }
        }

        #region ContextMenu
        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            builder.AddItem(MENU_ITEM_NAME, 0)
                .WithHeader(Properties.Resources.qrCodeMenuItem);
        }

        public void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            if (MENU_ITEM_NAME == name)
            {
                foreach (var obj in context.SelectedObjects)
                    CreateOrReplaceQrCode(obj);
            }
        }

        public void OnError(Exception error) { }
        public void OnCompleted() { }
        #endregion
    }
}