﻿/*
  Copyright © 2025 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.IO;
using System.Drawing;
using System.Windows.Input;
using System.Windows.Media.Imaging;
using Microsoft.Win32;
using Ascon.Pilot.SDK.Controls.Commands;
using PdfStamper.Models;

namespace PdfStamper.ViewModels
{
    public class SignatureSettingViewModel : BaseSettingViewModel, IDisposable
    {
        private readonly ISettingListener _settingListener;
        private readonly DelegateCommand _selectImageCommand;
        private byte[] _icon;
        private Bitmap _image;
        private string _stampHeader;

        public SignatureSettingViewModel(ISettingListener settingListener, SignatureSettingModel model = null) : base(settingListener)
        {
            _settingListener = settingListener;
            _selectImageCommand = new DelegateCommand(ShowDialog);
            
            if (model == null)
                SetDefaults();
            else
                FillWithData(model);
        }

        private void FillWithData(SignatureSettingModel model)
        {
            FillBaseWithData(model);
            SetIcon(model.Image);
            _stampHeader = model.StampHeader;
        }

        private void SetDefaults()
        {
            var defaultModel = SignatureSettingModel.GetDefault();
            SetBaseDefaults(defaultModel);
            _stampHeader = defaultModel.StampHeader;
        }

        private void ShowDialog()
        {
            var dialog = new OpenFileDialog
            {
                DefaultExt = ".png",
                Filter = "Image files (*.png;*.jpeg;*.jpg)|*.png;*.jpeg;*.jpg|JPEG Files (*.jpeg)|*.jpeg|PNG Files (*.png)|*.png|JPG Files (*.jpg)|*.jpg"
            };

            if (dialog.ShowDialog() == true)
            {
                SetIcon(File.ReadAllBytes(dialog.FileName));
                _settingListener.OnSettingsChanged();
            }
        }

        private void SetIcon(byte[] icon)
        {
            _icon = icon;
            
            if (icon == null)
                return;

            using (var stream = new MemoryStream(Icon))
            {
                Image = new Bitmap(stream);
            }
        }

        public ICommand SelectImageCommand => _selectImageCommand;

        public byte[] Icon
        {
            get => _icon;
            set
            {
                _icon = value;
                NotifyOfPropertyChange(nameof(Icon));
                _settingListener.OnSettingsChanged();
            }
        }

        public Bitmap Image
        {
            get => _image;
            set
            {
                _image?.Dispose();
                _image = value;
                NotifyOfPropertyChange(nameof(Thumbnail));
            }
        }

        public BitmapImage Thumbnail
        {
            get
            {
                if (Image == null)
                    return null;

                using (var memory = new MemoryStream())
                {
                    Image.Save(memory, System.Drawing.Imaging.ImageFormat.Png);
                    memory.Position = 0;
                    var bitmapImage = new BitmapImage();
                    bitmapImage.BeginInit();
                    bitmapImage.StreamSource = memory;
                    bitmapImage.CacheOption = BitmapCacheOption.OnLoad;
                    bitmapImage.EndInit();
                    bitmapImage.Freeze();

                    return bitmapImage;
                }
            }
        }

        public string StampHeader
        {
            get => _stampHeader;
            set
            {
                _stampHeader = value;
                NotifyOfPropertyChange(nameof(StampHeader));
                _settingListener.OnSettingsChanged();
            }
        }

        public override string ToString()
        {
            return Properties.Resources.SignatureStampTabItemHeader;
        }

        public void Dispose()
        {
            _image?.Dispose();
            _image = null;
        }
    }
}