/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reactive.Disposables;
using System.Reactive.Linq;
using System.Windows.Threading;
using Ascon.Pilot.SDK;
using DynamicData;

namespace ShowObjectsTree.DynamicData
{
    public class DataObjectService : IDataObjectService
    {
        private readonly IObjectsRepository _repository;
        private readonly CompositeDisposable _cleanUp = new CompositeDisposable();
        private readonly SourceCache<DataObject, Guid> _dataObjects = new SourceCache<DataObject, Guid>(x => x.Id);

        public IObservableCache<DataObject, Guid> DataObjects => _dataObjects.AsObservableCache();

        public DataObjectService(IObjectsRepository repository)
        {
            _repository = repository;
            _cleanUp.Add(_dataObjects);
        }

        public void Load(IEnumerable<Guid> ids)
        {
            var disposable = _repository.SubscribeObjects(ids)
                .Where(x => x.State == DataState.Loaded)
                .SubscribeOnDispatcher(DispatcherPriority.Background)
                .Subscribe(o =>
                {
                    var dataObject = new DataObject(o.Id, o.ParentId, o.DisplayName, o.Type.SvgIcon, o.Type.Sort, o.Children);
                    _dataObjects.AddOrUpdate(dataObject);
                });
            _cleanUp.Add(disposable);
        }

        public void Dispose()
        {
            _cleanUp.Dispose();
        }
    }
}
