/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Threading.Tasks;
using Ascon.Pilot.Common;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.DataModifier;
using Ascon.Pilot.ServerExtensions.SDK;
using Newtonsoft.Json.Linq;
using Ascon.Pilot.SDK.Common.WorkingDaysCalendar;

namespace ReassignedApprovalTaskDeadline
{
    [Export(typeof(IServerActivity))]
    public class ReassignedApprovalTaskDeadline : IServerActivity
    {
        private string _deadlineAttributeName;
        private string _reapprovalDaysAttributeName;
        private Dictionary<string, List<string>> _completionStateConfig;
        private IWorkingCalendarManager _workingCalendarManager;
        private const string DeadlineAttributeNameParam = "deadlineAttributeName";
        private const string ReapprovalDaysAttributeNameParam = "reapprovalDaysAttributeName";
        private const string CompletionStateConfig = "completionStateConfig";

        [ImportingConstructor]
        public ReassignedApprovalTaskDeadline(IWorkingCalendarManager workingCalendarManager)
        {
            _workingCalendarManager = workingCalendarManager;
        }

        public string Name => "MoveReassignedApprovalTaskDeadline";

        public async Task RunAsync(IModifierBase modifier, IModifierBackend backend,
            IServerActivityContext serverActivityContext,
            IAutomationEventContext automationEventContext)
        {
            if (serverActivityContext.Params.TryGetValue(DeadlineAttributeNameParam, out var deadlineAttributeName))
                _deadlineAttributeName = deadlineAttributeName.ToString();
            else
            {
                return;
            }

            if (serverActivityContext.Params.TryGetValue(ReapprovalDaysAttributeNameParam,
                    out var reapprovalDaysAttributeName))
                _reapprovalDaysAttributeName = reapprovalDaysAttributeName.ToString();
            else
            {
                return;
            }

            if (serverActivityContext.Params.TryGetValue(CompletionStateConfig, out object completionStateConfig))
                _completionStateConfig = GetSetOf<string>(completionStateConfig);
            else
            {
                return;
            }

            var document = automationEventContext.Source;
            // проверим, что в предыдущей версии документа были запросы на подпись

            var prevVersion = document.PreviousFileSnapshots.LastOrDefault();
            if(prevVersion == null)
                return;

            if (!prevVersion.Files.Any(x => x.SignatureRequests.Any()))
                return;

            IReadOnlyDictionary<int, INType> types = null;
            IReadOnlyList<INUserState> states = null;

            foreach (var dRelation in document.Relations.Where(x => x.Type == RelationType.TaskAttachments))
            {
                if (types == null)
                {
                    types = backend.GetTypes();
                    states = backend.GetMetadata().UserStates;
                }

                var task = backend.GetObject(dRelation.TargetId);
                await MoveDeadlineRecursivelyAsync(task, types, backend, modifier, states);
            }
        }

        private async Task MoveDeadlineRecursivelyAsync(INObject obj, IReadOnlyDictionary<int, INType> types,
            IModifierBackend backend, IModifierBase modifierBase, IReadOnlyList<INUserState> states)
        {

            await MoveDeadlineAsync(obj, types, modifierBase, states);

            foreach (var dChild in obj.Children)
            {
                var childTask = backend.GetObject(dChild.ObjectId);
                var type = types[childTask.TypeId];
                if (!type.IsTaskType())
                    continue;
                await MoveDeadlineRecursivelyAsync(childTask, types, backend, modifierBase, states);
            }
        }

        private async Task MoveDeadlineAsync(INObject obj, IReadOnlyDictionary<int, INType> types, IModifierBase modifierBase,
            IReadOnlyList<INUserState> states)
        {
            var type = types[obj.TypeId];
            if (!type.IsTaskType())
                return;
            if (!obj.Attributes.TryGetValue(_deadlineAttributeName, out var deadlineDate))
                return;
            if (!obj.Attributes.TryGetValue(_reapprovalDaysAttributeName, out var reapprovalDaysCount))
                return;
            if (!obj.Attributes.TryGetValue(SystemTaskAttributes.EXECUTOR_POSITION, out var executor))
                return;

            if (_completionStateConfig.TryGetValue(type.Name, out var completionStates))
            {
                var stateId = obj.GetNextTaskState();
                var state = states.FirstOrDefault(x => x.Id == stateId);
                if (state != null && completionStates.Contains(state.Name))
                    return;
            }

            var currentValue = deadlineDate.DateValue;
            if (currentValue.HasValue && currentValue.Value.Date != DateTime.MaxValue.Date)
            {
                if (reapprovalDaysCount.IntValue != null && executor.ArrayIntValue.Any())
                {
                    var newValue = await _workingCalendarManager.GetOrgUnitWorkingDateAsync(DateTime.Now, (int)reapprovalDaysCount.IntValue.Value, executor.ArrayIntValue[0]);
                    modifierBase.EditObject(obj).SetAttribute(_deadlineAttributeName, newValue);
                }
            }
        }


        public static Dictionary<string, List<V>> GetSetOf<V>(object var)
        {
            var jobj = JObject.FromObject(var).ToObject<Dictionary<string, object>>();

            var result = new Dictionary<string, List<V>>();

            if (jobj != null)
            {
                foreach (var key in jobj.Keys)
                {
                    var value = JArray.FromObject(jobj[key]).ToObject<List<V>>();
                    result.Add(key, value);
                }
            }

            return result;
        }

    }
}
