/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System.ComponentModel.Composition;
using System.Linq;
using System.Text;
using Ascon.Pilot.Common;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.DataModifier;
using Ascon.Pilot.ServerExtensions.SDK;

namespace UserActionLoggerExtension
{
    [Export(typeof(IUserActionAppender))]
    public class AddAdditionalInfoToLog : IUserActionAppender
    {
        public string ProcessEvent(IModifierBackend backend, IBaseEventContext context)
        {
            if (context is IPositionChangedContext positionChangedContext)
            {
                string name = "admin";
                var person = backend.CurrentPerson();
                if (person != null)
                    name = person.DisplayName;
                return $"{name} changed {positionChangedContext.Updated.Title}";
            }

            if (context is IPersonPositionChangedContext personPositionChangedContext)
            {
                if (!personPositionChangedContext.PersonOnPositionData.Person.HasValue) 
                    return string.Empty;

                var position = backend.GetOrganisationUnits()[personPositionChangedContext.PersonOnPositionData.PositionId];
                var person = backend.GetPerson(personPositionChangedContext.PersonOnPositionData.Person.Value);

                return personPositionChangedContext.PersonOnPositionData.IsRemoving 
                    ? $"{person.DisplayName} was removed from the position {position.Title} by {context.Login}" 
                    : $"{person.DisplayName} was appointed to the position {position.Title} by {context.Login}";
            }

            if (context is IChangesetContext changesetContext)
            {
                if (context.EventKind != EventKind.RelationChanging)
                    return string.Empty;
                var obj = backend.GetObject(changesetContext.ObjectId);
                var relations = changesetContext.Changes.Where(x =>
                    x.New.Id == obj.Id && x.Old != null && !x.New.Relations.SequenceEqual(x.Old.Relations));
                var type = backend.GetTypes()[obj.TypeId];

                var result = new StringBuilder();
                foreach (var relation in relations)
                {
                    var newRelations = relation.New.Relations.Except(relation.Old.Relations);
                    foreach (var newRelation in newRelations)
                    {
                        result.Append($"Type: {type.Name}, Title: {obj.GetTitle(type)} - relation added to ");
                        var relatedObj = backend.GetObject(newRelation.TargetId);
                        result.Append(relatedObj.GetTitle(backend.GetType(relatedObj.TypeId)));
                    }
                    var oldRelations = relation.Old.Relations.Except(relation.New.Relations);

                    foreach (var oldRelation in oldRelations)
                    {
                        result.Append($"Type: {type.Name}, Title: {obj.GetTitle(type)} - relation removed from ");
                        var relatedObj = backend.GetObject(oldRelation.TargetId);
                        result.Append(relatedObj.GetTitle(backend.GetType(relatedObj.TypeId)));
                    }
                }
                return result.ToString();
            }

            if (context is ILicenseContext licenseContext)
            {
                var result = new StringBuilder();

                result.Append(licenseContext.EventKind.ToString());
                result.Append(" of type id: ");
                result.Append(licenseContext.LicenseType.ToString());
                result.Append(" by ");
                result.Append(backend.GetPerson(licenseContext.PersonId).DisplayName);
                result.Append(" at ");
                result.Append(licenseContext.ServerDateTime.ToLocalTime().ToString("g"));
                result.Append(" from ip: ");
                result.Append(licenseContext.IpAddress);

                return result.ToString();
            }

            if (context is IFileContext fileContext)
            {
                var result = new StringBuilder();

                result.Append(context.EventKind.ToString());
                result.Append($" of file id {fileContext.FileId} ");
                result.Append(" from object ");
                result.Append(fileContext.ObjectTitle);
                result.Append(" of type name ");
                result.Append(backend.GetType(fileContext.TypeId).Name);
                result.Append(" at ");
                result.Append(fileContext.ServerDateTime.ToLocalTime().ToString("g"));
                result.Append(" from ip: ");
                result.Append(fileContext.IpAddress);
                result.AppendLine($" by {backend.GetPerson(fileContext.PersonId).DisplayName}");

                return result.ToString();
            }

            if (context is ISnapshotContext snapshotContext)
            {
                var result = new StringBuilder();

                result.Append(context.EventKind.ToString());
                result.Append($" of {snapshotContext.DeletedSnapshotsCount} snapshots");
                result.Append(" from object id ");
                result.Append(snapshotContext.ObjectId.ToString());
                result.Append(" at ");
                result.Append(context.ServerDateTime.ToLocalTime().ToString("g"));
                result.Append(" from ip: ");
                result.Append(context.IpAddress);

                return result.ToString();
            }

            if (context is IAuthenticationContext authenticationContext)
            {
                var result = new StringBuilder();

                result.Append(authenticationContext.EventKind.ToString());
                result.Append(" with login ");
                result.Append(authenticationContext.Login);
                result.Append(" with message: ");
                result.Append(authenticationContext.Message);
                result.Append(" at ");
                result.Append(authenticationContext.ServerDateTime.ToLocalTime().ToString("g"));
                result.Append(" from ip: ");
                result.Append(authenticationContext.IpAddress);

                return result.ToString();
            }

            if (context is IMetadataContext metadataContext)
            {
                var result = new StringBuilder();

                result.Append(metadataContext.EventKind.ToString());
                result.Append(" with login ");
                result.Append(metadataContext.Login);
                result.Append(" at ");
                result.Append(metadataContext.ServerDateTime.ToLocalTime().ToString("g"));
                result.Append(" from ip: ");
                result.Append(metadataContext.IpAddress);

                return result.ToString();
            }

            if (context is IPersonChangedContext personChangedContext)
            {
                var result = new StringBuilder();

                var loginField = personChangedContext.PersonChangeData.FirstOrDefault(x => x.FieldName == nameof(INPerson.Login));
                var isNewPerson = loginField != null && string.IsNullOrEmpty(loginField.OldValue);

                if (isNewPerson)
                {
                    result.Append($"\"{personChangedContext.Login}\" created person with data: ");
                    for (var i = 0; i < personChangedContext.PersonChangeData.Count; i++)
                    {
                        var data = personChangedContext.PersonChangeData[i];
                        result.Append($"{i+1}) Field \"{data.FieldName}\", value \"{data.NewValue}\" ");
                    }

                    return result.ToString();
                }

                var isDeletedField = personChangedContext.PersonChangeData.FirstOrDefault(x => x.FieldName == nameof(INPerson.IsDeleted));
                var isPersonDeleted = isDeletedField != null && bool.Parse(isDeletedField.NewValue);
                var changedPerson = backend.GetPerson(personChangedContext.PersonId).Login;

                if (isPersonDeleted)
                    return result.Append(
                        $"Person with login \"{changedPerson}\"" +
                        $" was deleted by \"{personChangedContext.Login}\"").ToString();

                result.Append($"\"{personChangedContext.Login}\" changed person \"{changedPerson}\". ");

                for (var i = 0; i < personChangedContext.PersonChangeData.Count; i++)
                {
                    var data = personChangedContext.PersonChangeData[i];
                    result.Append($"{i+1}) Changed field: \"{data.FieldName}\", old value: \"{data.OldValue}\", new value: \"{data.NewValue}\" ");
                }

                return result.ToString();
            }

            return null;
        }
    }
}
