/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Xps.Packaging;
using Ascon.Pilot.Common;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.DataModifier;

namespace Pilot.Xps.Domain.Signatures
{
    internal interface IPilotSignatureProxy
    {
        bool CanSign(INObject obj, XpsSignatureDefinition definition);
        bool CanSign(INObject obj, SignatureDisplayParams displayParam);
        bool CanSign(INObject obj, Guid? spotId, IEnumerable<int> positionIds);
        Guid? GetSignerId(INObject obj, XpsSignatureDefinition signatureDefinition);
        Guid? GetSignerId(IList<INFile> files, XpsSignatureDefinition signatureDefinition);
        string GetPersonTitleForDefinition(INObject obj, XpsSignatureDefinition signatureDefinition);
        string GetPersonTitleForDefinition(IList<INFile> files, XpsSignatureDefinition signatureDefinition);
        string GetPersonTitleForSignature(INObject obj, XpsSignatureDefinition signatureDefinition, string signerName, string position);
        string GetPersonTitleForSignature(IList<INFile> files, XpsSignatureDefinition signatureDefinition, string signerName, string position);
        string GetReadablePersonTitle(INObject obj, XpsSignatureDefinition signatureDefinition, string signerName, string position, bool isSigned);
        bool CanSign(IList<INFile> files, XpsSignatureDefinition toXpsSignatureDefinition);
    }

    class PilotSignatureProxy : IPilotSignatureProxy
    {
        private readonly IModifierBackend _backend;

        public PilotSignatureProxy(IModifierBackend backend)
        {
            _backend = backend ?? throw new ArgumentNullException(nameof(backend));
        }

        public bool CanSign(INObject obj, XpsSignatureDefinition definition)
        {
            return CanSign(obj, definition.SpotId);
        }

        public bool CanSign(INObject obj, SignatureDisplayParams displayParam)
        {
            return CanSign(obj, displayParam.SignId) && !displayParam.IsSigned;
        }

        public bool CanSign(IList<INFile> files, XpsSignatureDefinition definition)
        {
            return CanSign(files, definition.SpotId);
        }

        public Guid? GetSignerId(INObject obj, XpsSignatureDefinition signatureDefinition)
        {
            if (signatureDefinition?.SpotId == null)
                return null;

            var signature = GetSignature(signatureDefinition.SpotId.Value, obj);
            if (signature == null || signature.DatabaseId != _backend.GetDatabaseId())
                return null;

            var organizationUnit = _backend.GetOrganisationUnit(signature.PositionId);
            return XpsSignatureUtils.Id2Guid(_backend.GetDatabaseId(), organizationUnit.Id);
        }

        public Guid? GetSignerId(IList<INFile> files, XpsSignatureDefinition signatureDefinition)
        {
            if (signatureDefinition?.SpotId == null)
                return null;

            var signature = GetSignature(signatureDefinition.SpotId.Value, files);
            if (signature == null || signature.DatabaseId != _backend.GetDatabaseId())
                return null;

            var organizationUnit = _backend.GetOrganisationUnit(signature.PositionId);
            return XpsSignatureUtils.Id2Guid(_backend.GetDatabaseId(), organizationUnit.Id);
        }

        public string GetPersonTitleForDefinition(INObject obj, XpsSignatureDefinition signatureDefinition)
        {
            return GetReadablePersonTitle(obj.ActualFileSnapshot.Files.ToList(), signatureDefinition, signatureDefinition.RequestedSigner, "", false);
        }

        public string GetPersonTitleForDefinition(IList<INFile> files, XpsSignatureDefinition signatureDefinition)
        {
            return GetReadablePersonTitle(files, signatureDefinition, signatureDefinition.RequestedSigner, "", false);
        }

        public string GetPersonTitleForSignature(INObject obj, XpsSignatureDefinition signatureDefinition, string signerName, string position)
        {
            return GetReadablePersonTitle(obj.ActualFileSnapshot.Files.ToList(), signatureDefinition, signerName, position, true);
        }

        public string GetPersonTitleForSignature(IList<INFile> files, XpsSignatureDefinition signatureDefinition, string signerName, string position)
        {
            return GetReadablePersonTitle(files, signatureDefinition, signerName, position, true);
        }

        public string GetReadablePersonTitle(INObject obj, XpsSignatureDefinition signatureDefinition, string signerName, string position, bool isSigned)
        {
            return GetReadablePersonTitle(obj.ActualFileSnapshot.Files.ToList(), signatureDefinition, signerName, position, isSigned);
        }

        private string GetReadablePersonTitle(IList<INFile> files, XpsSignatureDefinition signatureDefinition, string signerName, string position, bool isSigned)
        {
            if (signatureDefinition == null || signatureDefinition.SpotId == null)
                return OrganisationUtils.GetTitle(signerName, position);

            var signature = GetSignature(signatureDefinition.SpotId.Value, files);
            if (signature == null || signature.DatabaseId != _backend.GetDatabaseId())
                return OrganisationUtils.GetTitle(signerName, "Unknown organization unit");

            var organizationUnit = _backend.GetOrganisationUnit(signature.PositionId);
            if (organizationUnit != null && !organizationUnit.Equals(DOrganisationUnit.Null))
            {
                if (isSigned)
                {
                    var person = new DPerson { DisplayName = signerName };
                    if (organizationUnit.Kind == OrgUnitKind.Position)
                        return organizationUnit.GetTitle(person);
                    if (!string.IsNullOrEmpty(position))
                        return OrganisationUtils.GetTitle(signerName, position);
                    return OrganisationUtils.GetTitle(OrganisationUtils.GetActualDisplayName(person), organizationUnit.Title);
                }
                var personOnPosition = _backend.GetMainPersonOnPosition(organizationUnit.Id);
                return organizationUnit.GetTitle(personOnPosition);
            }
            return OrganisationUtils.GetTitle(signerName, "Unknown organization unit");
        }

        private bool CanSign(INObject obj, Guid? spotId)
        {
            if (spotId == null)
                return false;

            var signature = GetSignature(spotId.Value, obj);
            if (signature == null || signature.DatabaseId != _backend.GetDatabaseId())
                return false;

            if (signature.ObjectId != Guid.Empty && !IsRelatedTaskStarted(signature.ObjectId))
                return false;

            var currentPerson = _backend.CurrentPerson();
            return currentPerson.Positions.Contains(signature.PositionId);
        }

        private bool CanSign(IList<INFile> files, Guid? spotId)
        {
            if (spotId == null)
                return false;

            var signature = GetSignature(spotId.Value, files);
            if (signature == null || signature.DatabaseId != _backend.GetDatabaseId())
                return false;

            if (signature.ObjectId != Guid.Empty && !IsRelatedTaskStarted(signature.ObjectId))
                return false;

            var currentPerson = _backend.CurrentPerson();
            return currentPerson.Positions.Contains(signature.PositionId);
        }

        private INSignature GetSignature(Guid spotId, IList<INFile> files)
        {
            return files.SelectMany(f => f.Signatures).FirstOrDefault(x => x.Id == spotId);
        }

        public bool CanSign(INObject obj, Guid? spotId, IEnumerable<int> positionIds)
        {
            if (spotId == null)
                return false;
            if (positionIds == null)
                return false;

            var signature = GetSignature(spotId.Value, obj);
            if (signature == null || signature.DatabaseId != _backend.GetDatabaseId())
                return false;

            if (signature.ObjectId != Guid.Empty && !IsRelatedTaskStarted(signature.ObjectId))
                return false;

            return positionIds.Contains(signature.PositionId);
        }

        private INSignature GetSignature(Guid spotId, INObject obj)
        {
            return obj.ActualFileSnapshot.Files.SelectMany(f => f.Signatures).FirstOrDefault(x => x.Id == spotId);
        }

        private bool IsRelatedTaskStarted(Guid taskId)
        {
            var taskObject = _backend.GetObject(taskId);

            if (taskObject.Attributes.TryGetValue(SystemTaskAttributes.STATE, out var stateValue))
            {
                if (!stateValue.GuidValue.HasValue || stateValue.GuidValue.Value == SystemStates.TASK_REVOKED_STATE_ID)
                    return false;

                if (stateValue.GuidValue.Value != SystemStates.NONE_STATE_ID || AllowSigningNotStarted(taskObject))
                {
                    return true;
                }
            }
            return false;
        }

        private bool AllowSigningNotStarted(INObject taskObject)
        {
            var type = _backend.GetType(taskObject.TypeId);
            var attr = type.Attributes.FirstOrDefault(x =>
                x.Name == SystemAttributes.ALLOW_SIGNING_AT_NONE_STATE_NAME);
            return attr != null && attr.Configuration == "<True/>";
        }
    }
}
