/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Windows.Annotations;
using System.Windows.Xps.Packaging;
using Ascon.Pilot.Common;
using Ascon.Pilot.DataClasses;
using Pilot.Xps.Domain.Annotations;
using Pilot.Xps.Domain.Context;
using Pilot.Xps.Domain.GraphicLayer;
using Pilot.Xps.Domain.Signatures;

namespace Pilot.Xps.Domain
{
    public interface IXpsPartLoader
    {
        byte[] DownloadXpsBody();
        IList<DigitalSignature> DownloadSignatures(out IList<XpsSignatureDefinition> signatureDefinitions);
        IList<XpsSignatureDefinition> DownloadSignatureDefinitions();
        IList<INSignature> GetAdditionalDefinitions();
    }

    class XpsPartLoader : IXpsPartLoader
    {
        private readonly IEnumerable<INFile> _files;
        private readonly IFileLoader _fileLoader;

        public XpsPartLoader(IEnumerable<INFile> files, IFileLoader fileLoader)
        {
            _files = files;
            _fileLoader = fileLoader;
        }

        public byte[] DownloadXpsBody()
        {
            var xpsFile = _files.FirstOrDefault(f => FileExtensionHelper.IsXpsAlike(f.Name));
            if (xpsFile == null) 
                return null;

            var bytes = _fileLoader.Download(xpsFile.Id, xpsFile.Size);
            return bytes;
        }

        public IList<DigitalSignature> DownloadSignatures(out IList<XpsSignatureDefinition> signatureDefinitions)
        {
            signatureDefinitions = DownloadSignatureDefinitions();// new List<XpsSignatureDefinition>();

            var signatures = new List<DigitalSignature>();
            var digitalSignatures = _files.Where(file => FileExtensionHelper.IsDigitalSignatureFile(file.Name));
            foreach (var dsFile in digitalSignatures)
            {
                var bytes = _fileLoader.Download(dsFile.Id, dsFile.Size);
                using (var stream = new MemoryStream(bytes))
                {
                    var signature = DigitalSignatureSerializer.Deserialize(stream);
                    signatures.Add(signature);
                }
            }

            return signatures;
        }

        public IList<XpsSignatureDefinition> DownloadSignatureDefinitions()
        {
            var signatureDefinitions = new List<XpsSignatureDefinition>();

            var v = _files.FirstOrDefault(f => FileExtensionHelper.IsSupportedDocument(f.Name));
            var signatures = v != null
                ? _files.Where(f => f.Id == v.Id).SelectMany(f => f.Signatures).ToList()
                : _files.SelectMany(f => f.Signatures).ToList();

            var definitions = signatures.Where(s => !s.IsAdditional).Select(d => d.ToXpsSignatureDefinition());
            signatureDefinitions.AddRange(definitions);
            return signatureDefinitions;
        }

        public IList<INSignature> GetAdditionalDefinitions()
        {
            var signatureDefinitions = new List<INSignature>();

            var v = _files.FirstOrDefault(f => FileExtensionHelper.IsSupportedDocument(f.Name));
            var signatures = v != null
                ? _files.Where(f => f.Id == v.Id).SelectMany(f => f.Signatures).ToList()
                : _files.SelectMany(f => f.Signatures).ToList();

            var definitions = signatures.Where(s => s.IsAdditional);//.Select(d => d.ToXpsSignatureDefinition());
            signatureDefinitions.AddRange(definitions);
            return signatureDefinitions;

            //if (storageObject == null)
            //    return;
            //var v = storageObject.ActualFileSnapshot.Files.FirstOrDefault(f => FileExtensionHelper.IsSupportedDocument(f.Name));
            //var signatures = v != null
            //    ? storageObject.ActualFileSnapshot.Files.Where(f => f.Id == v.Id).SelectMany(f => f.Signatures).ToList()
            //    : storageObject.ActualFileSnapshot.Files.SelectMany(f => f.Signatures).ToList();

            //var definitions = signatures.Where(s => !s.IsAdditional).Select(d => d.ToXpsSignatureDefinition());
            //_storedSignatureDefifnitions.AddRange(definitions);

            //// Все дополнительные запросы на подпись.
            //_storedAdditionalSignatureDefinitions.AddRange(signatures.Where(s => s.IsAdditional));
        }

        public IList<Annotation> DownloadAnnotations()
        {
            var result = new List<Annotation>();
            var annotationFiles = _files.Where(f => FileExtensionHelper.IsAnnotationFile(f.Name)).ToList();
            foreach (var annotationFile in annotationFiles)
            {
                var bytes = _fileLoader.Download(annotationFile.Id, annotationFile.Size);
                using (var stream = new MemoryStream(bytes))
                {
                    var annotation = (Annotation)AnnotationSerializers.AnnotationSerializer.Deserialize(stream);
                    AnnotationsHelper.ChangeMicrosoftModificationTime(annotation);
                    result.Add(annotation);
                }
            }

            return result;
        }

        public AnnotationMessages DownloadAnnotationsMessages()
        {
            var result = new List<DAnnotationMessage>();
            var annotationMessageFiles = _files.Where(f => FileExtensionHelper.IsAnnotationMessageFile(f.Name)).ToList();
            foreach (var messageFile in annotationMessageFiles)
            {
                var bytes = _fileLoader.Download(messageFile.Id, messageFile.Size);
                using (var stream = new MemoryStream(bytes))
                {
                    var message = (DAnnotationMessage)AnnotationSerializers.MessageSerializer.Deserialize(stream);
                    result.Add(message);
                }
            }

            return new AnnotationMessages {Messages = result};
        }

        public string DownloadTextLabels()
        {
            var textLabelsFile = _files.FirstOrDefault(f => f.Name.Equals(Constants.TEXT_LABELS_DEFINITION));
            if (textLabelsFile == null)
                return null;

            var bytes = _fileLoader.Download(textLabelsFile.Id, textLabelsFile.Size);
            using (var stream = new MemoryStream(bytes))
            using (var reader = new StreamReader(stream))
            {
                var text = reader.ReadToEnd();
                return text;
            }
        }

        public string DownloadBarcode()
        {
            var barcodeFile = _files.FirstOrDefault(f => f.Name.Equals(Constants.BARCODE_DEFINITION));
            if (barcodeFile == null)
                return null;

            var bytes = _fileLoader.Download(barcodeFile.Id, barcodeFile.Size);
            using (var stream = new MemoryStream(bytes))
            using (var reader = new StreamReader(stream))
            {
                var text = reader.ReadToEnd();
                return text;
            }
        }

        public IList<GraphicLayerElement> DownloadGraphicLayerElements()
        {
            var result = new List<GraphicLayerElement>();
            var graphicLayerElementFiles = _files.Where(f => f.Name.Contains(Constants.GRAPHIC_LAYER_ELEMENT_DEFINITION)).ToList();
            foreach (var graphicElementDefFile in graphicLayerElementFiles)
            {
                var bytes = _fileLoader.Download(graphicElementDefFile.Id, graphicElementDefFile.Size);
                using (var stream = new MemoryStream(bytes))
                {
                    var graphicElement = GraphicLayerElement.Deserialize(stream);
                    if (graphicElement != null && result.All(g => g.ElementId != graphicElement.ElementId))
                    {
                        var contentFileName = graphicElement.GetContentFileName();
                        var graphicContentFile = _files.FirstOrDefault(f => f.Name.Equals(contentFileName));
                        if (graphicContentFile != null)
                        {
                            result.Add(graphicElement);

                            var contentBytes = _fileLoader.Download(graphicContentFile.Id, graphicContentFile.Size);
                            using (var contentStream = new MemoryStream(contentBytes))
                            {
                                var content = new GraphicElementContentFactory().MakeContent(graphicElement.ContentType, contentStream);
                                graphicElement.GraphicLayerElementContent = content;
                            }
                        }
                    }
                }
            }

            return result;
        }
    }
}
