/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Linq;
using System.Threading.Tasks;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.Server.Api.Contracts;
using PilotShare.Core;

namespace PilotShare.Server.Models
{
    public class LinkTask
    {
        private readonly Guid _linkId;
        private readonly IServerApi _serverApi;

        public LinkTask(Guid linkId, IServerApi serverApi)
        {
            _linkId = linkId;
            _serverApi = serverApi;
        }

        public Task<LinkDefinition> LoadDefinitionAsync()
        {
            return Task<LinkDefinition>.Factory.StartNew(LoadLinkDefinition);
        } 

        private LinkDefinition LoadLinkDefinition()
        {
            var obj = _serverApi.GetObjects(new[] { _linkId }).FirstOrDefault();
            if (obj == null)
                throw new PilotShareLinkException("Link not found");

            if (obj.StateInfo.State != ObjectState.Alive)
                throw new PilotShareLinkException("The link has been removed");

            try
            {
                var linkDefinition = GetLinkDefinition(obj);
                return linkDefinition;
            }
            catch
            {
                throw new PilotShareLinkException("Incorrect link description");
            }
        }

        private LinkDefinition GetLinkDefinition(DObject obj)
        {
            var author = obj.Attributes["author"].IntValue ?? 0;
            var authorPerson = author > 0 ? _serverApi.LoadPeopleByIds(new[] { (int)author }).FirstOrDefault() : null;
            return new LinkDefinition
            {
                Password = obj.Attributes.TryGetValue("password", out var pass) ? pass.StrValue : null,
                ValidThrough = obj.Attributes.TryGetValue("expiration_date", out var expDate) ? expDate.DateValue : null,
                SharedBy = authorPerson?.DisplayName ?? "unknown",
                Item = StructureSerializer.Deserialize(obj.Attributes["structure"])
            };
        }
    }
}
