import { AsyncPipe } from '@angular/common';
import { AfterViewInit, Component, ElementRef, Inject, inject, OnInit, Renderer2, ViewChild, ViewEncapsulation } from '@angular/core';
import { MatButtonModule } from '@angular/material/button';
import { MatTableModule } from '@angular/material/table';
import { MatTooltip } from '@angular/material/tooltip';
import { BimElementPanelContext, IThemeService, Theme } from '@pilotdev/pilot-web-sdk';
import { BehaviorSubject, Observable } from 'rxjs';
import { THEME_SERVICE } from '../../tokens';

@Component({
  selector: 'app-element-view',
  standalone: true,
  imports: [
    MatButtonModule,
    MatTableModule,
    MatTooltip,
    AsyncPipe
  ],
  templateUrl: './element-view.component.html',
  styleUrl: './element-view.component.scss',
  encapsulation: ViewEncapsulation.ShadowDom
})
export class ElementViewComponent implements OnInit, AfterViewInit {
  private readonly _selectedElement$: BehaviorSubject<PilotWeb3D.ModelElement | undefined> = new BehaviorSubject<PilotWeb3D.ModelElement | undefined>(undefined);

  private readonly _themeService: IThemeService;
  private readonly _renderer: Renderer2;
  private readonly _context: BimElementPanelContext;

  get selectedElement$(): Observable<PilotWeb3D.ModelElement | undefined> {
    return this._selectedElement$.asObservable();
  }

  @ViewChild('wrapperElement') wrapperDiv!: ElementRef<HTMLDivElement>;

  constructor() {
    this._themeService = inject(THEME_SERVICE);
    this._renderer = inject(Renderer2);

    this._context = inject(BimElementPanelContext);
  }

  ngOnInit(): void {
    const model = this._context.viewer.model;

    const selection = model.getSelection();

    if (selection.length === 0)
      return;

    const firstSelected = selection[0];
    if (!firstSelected || firstSelected && firstSelected.elementIds.length === 0)
      return;

    const modelPart = model.getModelPart(firstSelected.modelPartId);

    if (!modelPart)
      return;

    this._selectedElement$.next(modelPart.elementTree.getElement(firstSelected.elementIds[0]));
  }

  ngAfterViewInit(): void {
    this.subscribeThemeChanges();
  }

  onVisibilityToggle(selectedElement: PilotWeb3D.ModelElement): void {
    const model = this._context.viewer.model;

    const visibleElements = model.getVisibleElements();
    const isElementVisible = visibleElements.some(modelIds => modelIds.modelPartId === selectedElement.modelPartId && modelIds.elementIds.includes(selectedElement.id));

    if (isElementVisible)
      model.hide(selectedElement.id, selectedElement.modelPartId);
    else
      model.show(selectedElement.id, selectedElement.modelPartId);
  }

  private subscribeThemeChanges(): void {
    this._themeService.change.subscribe((theme) => {
      if (theme.theme === Theme.Dark) {
        this._renderer.addClass(this.wrapperDiv.nativeElement, 'theme-dark');
      } else {
        this._renderer.removeClass(this.wrapperDiv.nativeElement, 'theme-dark');
      }
    });
  }
}
