﻿using System;
using System.Linq;
using Prism.Commands;
using System.ComponentModel;
using System.Threading.Tasks;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Runtime.CompilerServices;
using Ascon.Pilot.SDK;
using Ascon.Pilot.Bim.SDK.Search;
using Ascon.Pilot.Bim.SDK.ModelViewer;
using Ascon.Pilot.Bim.Search.SDK.Builders;
using Ascon.Pilot.Bim.Search.SDK.Serialization;
using Ascon.Pilot.Bim.Search.SDK.ModelProperties;
using Ascon.Pilot.Bim.Search.SDK.SearchExpressions;
using Ascon.Pilot.Bim.SDK.BottomPanelTabSample.Tools;
using Ascon.Pilot.Bim.SDK.BottomPanelTabSample.Models;
using Ascon.Pilot.Bim.SDK.ModelTab.BottomPanel;

namespace Ascon.Pilot.Bim.SDK.BottomPanelTabSample
{
    public class BottomTabControlViewModel : ISearchController, INotifyPropertyChanged
    {
        private readonly IModelSearchManager _searchManager;
        private IModelSearchService _searchService;
        private IModelViewer _modelViewer;
        private bool _searchServiceIsLoaded;
        private readonly IObjectLoader _objectLoader;
        private readonly List<ModelPartSelection> _modelParts = new List<ModelPartSelection>();

        public BottomTabControlViewModel(IModelSearchManager searchManager, IObjectsRepository objectsRepository)
        {
            _searchManager = searchManager;
            _objectLoader = new ObjectLoader(objectsRepository);
            AddCommand = new DelegateCommand(Add);
            SearchCommand = new DelegateCommand(Search, CanSearch);
            RemoveCommand = new DelegateCommand<SearchSetViewModel>(Remove);
        }

        public IBottomPanelTab BottomTab { get; set; }

        public IModelViewer ModelViewer
        {
            get => _modelViewer;
            set
            {
                _modelViewer = value;
                OnPropertyChanged(nameof(ModelViewer));
            }
        }

        public bool SearchServiceIsLoaded
        {
            get => _searchServiceIsLoaded;
            set
            {
                _searchServiceIsLoaded = value;
                OnPropertyChanged(nameof(SearchServiceIsLoaded));
            }
        }

        public ObservableCollection<ISearchSetExpression> Expressions { get; set; } = new ObservableCollection<ISearchSetExpression>();
        public DelegateCommand AddCommand { get; }
        public DelegateCommand SearchCommand { get; }
        public DelegateCommand<SearchSetViewModel> RemoveCommand { get; }

        private void Add()
        {
            var newExpr = new SearchSetViewModel(_searchService, this);
            Expressions.Add(newExpr);
            InvalidateSearch();
        }

        private void Remove(ISearchSetExpression parameter)
        {
            Expressions.Remove(parameter);
            InvalidateSearch();
        }

        public IEnumerable<ModelPartSelection> GetModelParts()
        {
            return _modelParts;
        }

        public void InvalidateSearch()
        {
            SearchCommand.RaiseCanExecuteChanged();
        }

        private bool CanSearch()
        {
            return Expressions.Any() && Expressions.All(x => !string.IsNullOrEmpty(x.PropertyName)) && Expressions.All(x => x.IsValid());
        }

        private void Search()
        {
            var searchSetBuilder = GetSearchSetBuilder(Expressions);
            var res = _searchService.Search(searchSetBuilder.Build(), 1000).ToList();

            _modelViewer.ClearSelection();
            _modelViewer.Select(res);
            _modelViewer.IsolateSelection();
        }

        private static IModelSearchSetBuilder GetSearchSetBuilder(IEnumerable<ISearchSetExpression> expressions)
        {
            var builder = ModelSearchSetBuilder.CreateBuilder(SearchSetLogicalOperator.And);
            var orExpressions = new List<IModelSearchExpression>();

            foreach (var expression in expressions)
            {
                var modelExpression = ToSearchExpression(expression);

                if (expression.LogicalOperation == LogicalOperator.Or)
                {
                    orExpressions.Add(modelExpression);
                    continue;
                }

                if (orExpressions.Any())
                {
                    var expressionsBuilder = ModelSearchExpressionsBuilder.CreateBuilder();
                    orExpressions.ForEach(x => expressionsBuilder.AddExpression(x));

                    builder.NewGroup(SearchSetLogicalOperator.Or).AddExpressions(expressionsBuilder);
                    orExpressions.Clear();
                }

                if (expression.LogicalOperation == LogicalOperator.And)
                {
                    builder.AddExpressions(ModelSearchExpressionsBuilder.CreateBuilder().AddExpression(modelExpression));
                    continue;
                }

                throw new NotSupportedException(expression.LogicalOperation.ToString());
            }

            if (orExpressions.Any())
            {
                var expressionsBuilder = ModelSearchExpressionsBuilder.CreateBuilder();
                orExpressions.ForEach(x => expressionsBuilder.AddExpression(x));

                builder.NewGroup(SearchSetLogicalOperator.Or).AddExpressions(expressionsBuilder);
                orExpressions.Clear();
            }

            return builder;
        }

        private static IModelSearchExpression ToSearchExpression(ISearchSetExpression expression)
        {
            var categoryAndNameSplit = expression.PropertyName.Split(new[] {Constants.CATEGORY_NAME_SPLITTER}, StringSplitOptions.RemoveEmptyEntries);
            var (category, name) = new Tuple<string, string>(categoryAndNameSplit[0], categoryAndNameSplit[1]);

            if (expression.Operator == EqualityOperator.Defined)
                return ModelProperty.WithName(name, category, PropertyDataType.String).Defined();

            switch (expression.Kind)
            {
                case SearchExpressionPropertyKind.String:
                case SearchExpressionPropertyKind.Lookup:
                    switch (expression.Operator)
                    {
                        case EqualityOperator.Equal:
                            {
                                if (expression.Expression is ModelPartSelection selection)
                                    return ModelProperty.ModelPartId.EqualTo(selection.Id);

                                return Guid.TryParse(expression.Expression.ToString(), out var guidValue)
                                    ? ModelProperty.WithName(name, category, PropertyDataType.Guid).EqualTo(guidValue)
                                    : ModelProperty.WithName(name, category, PropertyDataType.String).EqualTo(expression.Expression.ToString());
                            }
                        default:
                            throw new NotSupportedException(expression.Operator.ToString());
                    }

                case SearchExpressionPropertyKind.Double:
                    var doubleValue = double.Parse(expression.Expression.ToString());
                    switch (expression.Operator)
                    {
                        case EqualityOperator.Equal:
                            return ModelProperty.WithName(name, category, PropertyDataType.Double).EqualTo(doubleValue);
                        case EqualityOperator.GreaterOrEqual:
                            return ModelProperty.WithName(name, category, PropertyDataType.Double).GreaterThanOrEqual(doubleValue);
                        case EqualityOperator.LessOrEqual:
                            return ModelProperty.WithName(name, category, PropertyDataType.Double).LessThanOrEqual(doubleValue);
                        default:
                            throw new NotSupportedException(expression.Operator.ToString());
                    }

                default:
                    throw new NotSupportedException(expression.ToString());
            }
        }

        private async Task LoadModelParts()
        {
            var modelParts = _searchService.Search(ModelSearchSetBuilder
                .CreateBuilder(SearchSetLogicalOperator.And)
                .AddExpressions(ModelSearchExpressionsBuilder
                    .CreateBuilder()
                    .AddExpression(ModelProperty.Type.EqualTo("IfcProject"))).Build(), 500);

            foreach (var element in modelParts)
            {
                var modelPartObj = await _objectLoader.Load(element.ModelPartId);
                _modelParts.Add(new ModelPartSelection(element.ModelPartId, modelPartObj.DisplayName));
            }
        }

        public void OnModelLoaded(IModelViewer modelViewer)
        {
            ModelViewer = modelViewer;
            BottomTab.IsVisible = ModelViewer != null;
            _searchService?.Dispose();
            _modelParts.Clear();

            Task.Run(async () =>
            {
                _searchService = await _searchManager.GetModelSearchServiceAsync(ModelViewer.ModelId);
                SearchServiceIsLoaded = _searchService != null;
                await LoadModelParts();
                return _searchService;
            });
        }

        public void OnModelClosed()
        {
            ModelViewer = null;
            BottomTab.IsVisible = false;
            _modelParts.Clear();
            _searchService?.Dispose();
        }

        #region NotifyPropertyChanged

        public event PropertyChangedEventHandler PropertyChanged;
        protected virtual void OnPropertyChanged([CallerMemberName] string propertyName = null)
        {
            PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(propertyName));
        }

        #endregion
    }
}