﻿/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System.Collections.Generic;
using System;
using System.ComponentModel.Composition;
using Ascon.Pilot.SDK;
using Newtonsoft.Json;
using BcfApiService.DTOs;
using BcfApiService.Services;

namespace BcfApiService
{
    [Export(typeof(IDataPlugin))]
    public class WebService : IDataPlugin
    {
        private readonly WebServer _application;
        private readonly ProjectService _projectService;
        private readonly TopicService _topicService;
        private readonly CommentService _commentService;

        [ImportingConstructor]
        public WebService(IObjectsRepository objectsRepository, ISearchService searchService, IMessagesRepository messagesRepository)
        {
            _application = new WebServer(3001);
            _projectService = new ProjectService(objectsRepository, searchService);
            _topicService = new TopicService(objectsRepository, searchService);
            _commentService = new CommentService(objectsRepository, searchService, messagesRepository);

            _application.GetMethod(Routes.GET_PROJECTS, async (WebServerRequest req, WebServerResponse res) =>
            {
                res.SetMIMEtype("application/json");
                IEnumerable<ProjectDTO> projects = await _projectService.GetProjects();
                string serializedProjects = JsonConvert.SerializeObject(projects);
                res.WriteResponse(serializedProjects);
            });

            _application.GetMethod(Routes.GET_PROJECT_BY_ID, async (WebServerRequest req, WebServerResponse res) =>
            {
                res.SetMIMEtype("application/json");
                Guid projectId = Guid.Parse(req.GetUrlAbsolutePath().Split('/')[4]);
                ProjectDTO project = await _projectService.FindProjectById(projectId);
                string serializedProject;
                if (project == null)
                    serializedProject = JsonConvert.SerializeObject(new object());
                else
                    serializedProject = JsonConvert.SerializeObject(project);
                res.WriteResponse(serializedProject);
            });

            _application.GetMethod(Routes.GET_TOPICS, async (WebServerRequest req, WebServerResponse res) =>
            {
                res.SetMIMEtype("application/json");
                Guid projectId = Guid.Parse(req.GetUrlAbsolutePath().Split('/')[4]);
                IEnumerable<TopicDTO> topics = await _topicService.GetTopics(projectId);
                string serializedTopics = JsonConvert.SerializeObject(topics);
                res.WriteResponse(serializedTopics);
            });

            _application.GetMethod(Routes.GET_TOPIC_BY_ID, async (WebServerRequest req, WebServerResponse res) =>
            {
                res.SetMIMEtype("application/json");
                Guid projectId = Guid.Parse(req.GetUrlAbsolutePath().Split('/')[4]);
                Guid topicId = Guid.Parse(req.GetUrlAbsolutePath().Split('/')[6]);
                TopicDTO topic = await _topicService.FindTopicById(projectId, topicId);
                string serializedTopic;
                if (topic == null)
                    serializedTopic = JsonConvert.SerializeObject(new object());
                else
                    serializedTopic = JsonConvert.SerializeObject(topic);
                res.WriteResponse(serializedTopic);
            });

            _application.GetMethod(Routes.GET_COMMENTS, async (WebServerRequest req, WebServerResponse res) =>
            {
                res.SetMIMEtype("application/json");
                Guid projectId = Guid.Parse(req.GetUrlAbsolutePath().Split('/')[4]);
                Guid topicId = Guid.Parse(req.GetUrlAbsolutePath().Split('/')[6]);
                IEnumerable<CommentDTO> comments = await _commentService.GetComments(projectId, topicId);
                
                // TODO: add orderby and filter!

                string serializedComments = JsonConvert.SerializeObject(comments);
                res.WriteResponse(serializedComments);
            });

            _application.GetMethod(Routes.GET_COMMENT_BY_ID, async (WebServerRequest req, WebServerResponse res) =>
            {
                res.SetMIMEtype("application/json");
                Guid projectId = Guid.Parse(req.GetUrlAbsolutePath().Split('/')[4]);
                Guid topicId = Guid.Parse(req.GetUrlAbsolutePath().Split('/')[6]);
                Guid commentId = Guid.Parse(req.GetUrlAbsolutePath().Split('/')[8]);
                CommentDTO comment = await _commentService.FindCommentById(projectId, topicId, commentId);
                string serializedComment;
                if (comment == null)
                    serializedComment = JsonConvert.SerializeObject(new object());
                else
                    serializedComment = JsonConvert.SerializeObject(comment);
                res.WriteResponse(serializedComment);
            });

            _application.Start();
        }

        public void Stop()
        {
            _application.Stop();
        }
    }
}
