/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.ComponentModel;
using System.ComponentModel.Composition;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Threading;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Menu;
using Ascon.Pilot.Theme.ColorScheme;
using Ascon.Pilot.Theme.Controls;
using ChangeFilesMonitor.Tools;

namespace ChangeFilesMonitor
{
    [Export(typeof(IMenu<SystemTrayContext>))]
    public class Main : IMenu<SystemTrayContext>, IHandle<UnloadedEventArgs>, IHandle<LoadedEventArgs>
    {
        private const string PLUGIN_TITLE = "Измененные файлы";
        private readonly IPilotStorageCommandController _pilotStorageCommandController;
        private Window _dialog;
        private MainViewModel _mainViewModel;
        private HotKey _toggleHotKey;
        private HotKey _sendAllHotKey;

        [ImportingConstructor]
        public Main(IEventAggregator eventAggregator, IPilotDialogService service, IPilotStorageCommandController pilotStorageCommandController)
        {
            _pilotStorageCommandController = pilotStorageCommandController;
            eventAggregator.Subscribe(this);
            InitColorScheme(service);

            Application.Current.Dispatcher.BeginInvoke(new Action(SubscribeHotKey));

        }

        private void ShowWindow()
        {
            if(_dialog != null)
                return;

            _dialog = CreateWindow();
            _dialog.Show();
        }

        private Window CreateWindow()
        {
            var view = new MainView();
            var dialog = new DialogWindow();

            var style = (Style)view.FindResource("DialogWindowStyle");
            dialog.Style = style;

            view.DataContext = _mainViewModel ?? (_mainViewModel = new MainViewModel(_pilotStorageCommandController));
            dialog.Content = view;
            dialog.ShowInTaskbar = true;
            dialog.MinWidth = 150;
            dialog.Width = 350;
            dialog.MinHeight= 200;
            dialog.Height = 400;
            dialog.Topmost = true;
            dialog.Title = PLUGIN_TITLE;

            dialog.Loaded += DialogOnLoaded;
            dialog.Closing += DialogOnClosing;
            return dialog;
        }

        private void DialogOnClosing(object sender, CancelEventArgs cancelEventArgs)
        {
            SaveState(sender);
            _dialog = null;
        }

        private static void SaveState(object sender)
        {
            var settingsPath = SpecialDirectoryProvider.SettingsFilename;
            var settings = new Tools.WindowState();

            var window = (Window) sender;
            settings.Left = window.Left;
            settings.Top = window.Top;
            settings.Width = window.Width;
            settings.Height = window.Height;

            SettingsStore.SaveWindowsPosition(settingsPath, settings);
        }

        private void DialogOnLoaded(object sender, RoutedEventArgs routedEventArgs)
        {
            var settingsPath = SpecialDirectoryProvider.SettingsFilename;
            var settings = SettingsStore.GetWindowsPosition(settingsPath);

            var window = (Window)sender;
            window.Left = settings.Left;
            window.Top = settings.Top;
            window.Width = settings.Width;
            window.Height = settings.Height;
        }

        private void InitColorScheme(IPilotDialogService pilotDialogService)
        {
            var accentColorObj = ColorConverter.ConvertFromString(pilotDialogService.AccentColor);
            if (accentColorObj == null)
                return;

            var accentColor = (Color)accentColorObj;
            ColorScheme.Initialize(accentColor, pilotDialogService.Theme);
        }

        public void Handle(UnloadedEventArgs message)
        {
            var dialog = _dialog;
            dialog?.Dispatcher.BeginInvoke(new Action(() =>
            {
                SaveState(dialog);
                dialog.Close();
            }));
            _toggleHotKey.Dispose();
            _sendAllHotKey.Dispose();
        }

        public void Handle(LoadedEventArgs message)
        {
            RunDialog();
        }

        private void RunDialog()
        {
            Application.Current.Dispatcher.BeginInvoke(new Action(ShowWindow));
        }

        public void Build(IMenuBuilder builder, SystemTrayContext context)
        {
            builder.AddItem("ChangeFilesMonitor", 0).WithHeader(PLUGIN_TITLE);
        }

        public void OnMenuItemClick(string name, SystemTrayContext context)
        {
            RunDialog();
        }

        private void SubscribeHotKey()
        {
            _toggleHotKey = new HotKey(ModifierKeys.Control | ModifierKeys.Shift, Keys.Oem3, Application.Current.MainWindow);
            _toggleHotKey.HotKeyPressed += OnToggleHotKeyPressed;

            _sendAllHotKey = new HotKey(ModifierKeys.Control | ModifierKeys.Shift, Keys.W, Application.Current.MainWindow);
            _sendAllHotKey.HotKeyPressed += OnSendAllHotKeyPressed;
        }

        private void OnToggleHotKeyPressed(HotKey obj)
        {
            Application.Current.Dispatcher.BeginInvoke(new Action(ToggleDialog));
        }

        private void OnSendAllHotKeyPressed(HotKey obj)
        {
            Application.Current.Dispatcher.BeginInvoke(new Action(SendAll));
        }

        private void ToggleDialog()
        {
            if(_dialog != null)
                _dialog.Close();
            else
                ShowWindow();
        }

        private void SendAll()
        {
            if(_mainViewModel == null)
                return;
            if(_mainViewModel.SendAllToServerCommand.CanExecute())
                _mainViewModel.SendAllToServerCommand.Execute();
        }
    }
}
