/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Extensions;

namespace CustomXpsAnnotations
{
    public interface IRemarkModifier
    {
        Task<IObjectBuilder> CreateRemark(IDataObject parent, string remarksFolderName, string remarkTypeName, XpsAnnotation annotation);
        void DeleteRemark(Guid remarkId);
        IObjectBuilder EditRemark(Guid remarkId);
        void Apply();
    }

    [Export(typeof(IRemarkModifier))]
    public class RemarkModifier : IRemarkModifier
    {
        private readonly IObjectModifier _modifier;
        private readonly IObjectsRepository _repository;

        [ImportingConstructor]
        public RemarkModifier(IObjectModifier modifier, IObjectsRepository repository)
        {
            _modifier = modifier;
            _repository = repository;
        }

        public async Task<IObjectBuilder> CreateRemark(IDataObject parent, string remarksFolderName, string remarkTypeName, XpsAnnotation annotation)
        {
            var remarkFolderType = _repository.GetType(remarksFolderName);
            var remarkType = _repository.GetType(remarkTypeName);
            var folders = parent.TypesByChildren.Where(x => x.Value == remarkFolderType.Id);
            IDataObject versionFolder = null;
            DateTime version = parent.ActualFileSnapshot.Created.ToUniversalTime();
            foreach (var folder in folders)
            {
                var folderObj = await _repository.GetObjectAsync(folder.Key, CancellationToken.None);
                if (IsRemarksFolderRelatedToSnapshot(folderObj, version))
                    versionFolder = folderObj;
            }

            var folderId = GetFolder(parent, versionFolder, version, remarkFolderType);

            IObjectBuilder builder = null;
            var serialized = XpsAnnotationSerializers.Serialize(annotation);
            var existingAnnotation = versionFolder?.Children.FirstOrDefault(x => x == annotation.AnnotationId);
            if (versionFolder == null || existingAnnotation.Value == Guid.Empty)
            {
                builder = _modifier.CreateById(annotation.AnnotationId, folderId, remarkType);
            }
            else
            {
                builder = _modifier.EditById(annotation.AnnotationId);
            }
            builder.SetAttribute(SystemAttributeNames.DOCUMENT_REMARK_ANNOTATION, serialized);
            builder.SetAttribute(SystemAttributeNames.DOCUMENT_REMARK_ANNOTATION + version.Ticks, 1);

            return builder;
        }

        public void DeleteRemark(Guid remarkId)
        {
            _modifier.DeleteById(remarkId);
            _modifier.Apply();
        }

        public IObjectBuilder EditRemark(Guid remarkId)
        {
            return _modifier.EditById(remarkId);
        }

        public void Apply()
        {
            _modifier.Apply();
        }

        private Guid CreateFolder(Guid actualObjId, DateTime version, IType folderType)
        {
            var folderId = Guid.NewGuid();
            var res = _modifier.CreateById(folderId, actualObjId, folderType);
            res.MakeSecret();
            res.SetAttribute(SystemAttributeNames.DOCUMENT_REMARK_VERSION, version);
            return folderId;
        }

        private bool IsRemarksFolderRelatedToSnapshot(IDataObject remarksFolder, DateTime version)
        {
            if (remarksFolder.Attributes.TryGetValue(SystemAttributeNames.DOCUMENT_REMARK_VERSION, out var created))
            {
                if (created is DateTime time && time.ToUniversalTime() == version)
                    return true;
            }
            return false;
        }

        private Guid GetFolder(IDataObject actualObject, IDataObject versionFolder, DateTime version, IType folderType)
        {
            var folder = versionFolder?.Id ?? CreateFolder(actualObject.Id, version, folderType);
            return folder;
        }
    }
}
