﻿/*
  Copyright © 2025 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Linq;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using Ascon.Pilot.SDK;
using Ascon.Pilot.Theme;
using Ascon.Pilot.SDK.Controls;
using Ascon.Pilot.Theme.Controls;
using Ascon.Pilot.SDK.Controls.Commands;
using Ascon.Pilot.SDK.Controls.Converters;
using PdfStamper.Models;
using PdfStamper.Serializing;
using PdfStamper.SettingsToolbar;

namespace PdfStamper.ViewModels
{
    public interface ISettingListener
    {
        void OnSettingsChanged();
    }

    internal class SettingsViewModel : PropertyChangedBase, ISettingListener, IDisposable
    {
        private readonly ISettingValueProvider _settingValueProvider;
        private IEnumerable<IType> _documentTypes;
        private IType _selectedSettingType;
        private Dictionary<string, List<BaseSettingModel>> _settings = new Dictionary<string, List<BaseSettingModel>>();

        public SettingsViewModel(ISettingValueProvider settingValueProvider, IObjectsRepository repository)
        {
            _settingValueProvider = settingValueProvider;
            _documentTypes = repository.GetTypes().Where(x => !x.IsService && !x.IsDeleted && x.Kind != TypeKind.System && x.HasFiles);
            DeleteSetting = new DelegateCommand(DeleteSelectedSetting, CanDeleteSelectedSetting);

            Initialize();
            ReFillToolbar();
        }

        private void Initialize()
        {
            var currentValue = _settingValueProvider.GetValue();
            if (currentValue == null)
                return;

            _settings = Serializer.DeserializeSettings(currentValue);

            foreach (var keyValuePair in _settings)
                SettingTypes.Add(_documentTypes.First(x => x.Name.Equals(keyValuePair.Key)));
        }

        private void ReFillTabControl(IType type)
        {
            ClearStampsSettings();

            if (type == null) 
                return;

            SignatureSettingViewModel signatureSetting = null;
            IntoProductionSettingViewModel intoProductionSetting = null;
            if (_settings.TryGetValue(type.Name, out var currentModels))
            {
                foreach (var model in currentModels)
                {
                    if (model is SignatureSettingModel signatureSettingModel)
                        signatureSetting = new SignatureSettingViewModel(this, signatureSettingModel);
                    else if (model is IntoProductionSettingModel intoProductionSettingModel)
                        intoProductionSetting = new IntoProductionSettingViewModel(this, intoProductionSettingModel);
                }
            }
            else
            {
                signatureSetting = new SignatureSettingViewModel(this);
                intoProductionSetting = new IntoProductionSettingViewModel(this);
            }

            StampsSettings.Add(signatureSetting);
            StampsSettings.Add(intoProductionSetting);
        }

        private void CreateSetting(IType type)
        {
            SettingTypes.Add(type);
            SelectedSettingType = type;

            ReFillToolbar();
            OnSettingsChanged();
        }

        private void DeleteSelectedSetting()
        {
            ClearStampsSettings();
            
            _settings.Remove(SelectedSettingType.Name);
            SettingTypes.Remove(SelectedSettingType);
            SelectedSettingType = null;
            OnSettingsChanged();
            ReFillToolbar();
        }

        private void ClearStampsSettings()
        {
            foreach (var setting in StampsSettings)
            {
                if (setting is IDisposable disposable)
                    disposable.Dispose();
            }
            StampsSettings.Clear();
        }

        private bool CanDeleteSelectedSetting()
        {
            return SelectedSettingType != null;
        }

        private void ReFillToolbar()
        {
            ToolbarItems.Clear();
            var types = _documentTypes.Where(x => !SettingTypes.Select(y => y.Name).Contains(x.Name));

            var buttons = new List<IToolbarButtonItem>();
            foreach (var type in types)
            {
                var item = new ToolbarButtonItem(type.Name)
                {
                    Header = type.Title,
                    Icon = ByteImageConverter.Convert(type.SvgIcon),
                    CommandParameter = type,
                    Command = new DelegateCommand<IType>(CreateSetting)
                };
                buttons.Add(item);
            }

            ToolbarItems.Add(new ToolbarMenuButtonItem("tbiCreate", buttons) { Header = Properties.Resources.CreateCommandHeader, Icon = Icons.Instance.CreateIcon });
            ToolbarItems.Add(new ToolbarButtonItem("tbiDeleteItem") { Header = Properties.Resources.RemoveCommandHeader, Icon = Icons.Instance.RemoveIcon, Command = DeleteSetting });
        }

        public DelegateCommand DeleteSetting { get; }

        public ObservableCollection<IToolbarButtonItem> ToolbarItems { get; } = new ObservableCollection<IToolbarButtonItem>();

        public ObservableCollection<IType> SettingTypes { get; } = new ObservableCollection<IType>();

        public ObservableCollection<BaseSettingViewModel> StampsSettings { get; } = new ObservableCollection<BaseSettingViewModel>();

        public IType SelectedSettingType
        {
            get => _selectedSettingType;
            set
            {
                if (ReferenceEquals(_selectedSettingType, value))
                    return;

                _selectedSettingType = value;
                ReFillTabControl(value);

                NotifyOfPropertyChange(nameof(SelectedSettingType));
                DeleteSetting.RaiseCanExecuteChanged();
            }
        }

        public void OnSettingsChanged()
        {
            if (SelectedSettingType != null)
            {
                var selectedSettings = new List<BaseSettingModel>();
                foreach (var setting in StampsSettings)
                {
                    if (setting is SignatureSettingViewModel sVm)
                    {
                        var signModel = new SignatureSettingModel
                        {
                            Image = sVm.Icon,
                            StampHeader = sVm.StampHeader,
                            Position = sVm.SelectedPosition,
                            OffsetX = int.Parse(sVm.XOffsetStr),
                            OffsetY = int.Parse(sVm.YOffsetStr),
                            IsOnLastPage = sVm.IsOnLastPage,
                            PageNumber = int.Parse(sVm.PageNumber),
                            OffsetPreviousX = int.Parse(sVm.XOffsetStrPrevious),
                            OffsetPreviousY = int.Parse(sVm.YOffsetStrPrevious),
                            PutStamp = sVm.PutStamp
                        };
                        selectedSettings.Add(signModel);
                    }
                    else if (setting is IntoProductionSettingViewModel pVm)
                    {
                        var productionModel = new IntoProductionSettingModel
                        {
                            ProductionRole = pVm.ProductionRole,
                            Position = pVm.SelectedPosition,
                            OffsetX = int.Parse(pVm.XOffsetStr),
                            OffsetY = int.Parse(pVm.YOffsetStr),
                            IsOnLastPage = pVm.IsOnLastPage,
                            PageNumber = int.Parse(pVm.PageNumber),
                            OffsetPreviousX = int.Parse(pVm.XOffsetStrPrevious),
                            OffsetPreviousY = int.Parse(pVm.YOffsetStrPrevious),
                            PutStamp = pVm.PutStamp
                        };
                        selectedSettings.Add(productionModel);
                    }
                }

                _settings[SelectedSettingType.Name] = selectedSettings;
            }

            _settingValueProvider.SetValue(Serializer.SerializeSettings(_settings));
        }

        public void Dispose()
        {
            _documentTypes = Enumerable.Empty<IType>();
            ToolbarItems.Clear();
            ClearStampsSettings();
        }
    }
}