/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using Ascon.Pilot.SDK;
using DynamicData;
using Moq;
using NUnit.Framework;
using ProjectCloneWizard.Common;
using ProjectCloneWizard.Data;
using ProjectCloneWizard.ViewModels;

namespace ProjectCloneWizard.Tests
{
    [TestFixture]
    class SelectProjectTemplatePageViewModelTests
    {
        private Mock<IDataObjectService> _dataObjectServiceMock;

        [SetUp]
        public void SetUp()
        {
            _dataObjectServiceMock = new Mock<IDataObjectService>();
            var dataObjects = new SourceCache<IDataObject, Guid>(x => x.Id);
            _dataObjectServiceMock.Setup(s => s.DataObjects).Returns(dataObjects.AsObservableCache);
        }

        [Test]
        public void should_return_correct_values()
        {
            var creationInfo = new CreationInfo();
            creationInfo.Parent = TestTools.GetIDataObjectMock(Guid.NewGuid()).Object;
            var viewModel = new SelectProjectTemplatePageViewModel(creationInfo, _dataObjectServiceMock.Object);

            var expected = "Шаг 1 из 4. Выберите элемент для использования в качестве шаблона";
            var actual = viewModel.GetTitle();
            Assert.AreEqual(expected, actual);

            expected = "";
            actual = viewModel.GetTemplateName();
            Assert.AreEqual(expected, actual);

            actual = viewModel.NextButtonCaption();
            Assert.IsNull(actual);

            actual = viewModel.NextButtonCaption();
            Assert.IsNull(actual);

            Assert.IsFalse(viewModel.CanGoBack());
            Assert.IsFalse(viewModel.CanGoNext());
        }

        [Test]
        public void should_load_tree()
        {
            //given
            var creationInfo = new CreationInfo();
            var viewModel = new SelectProjectTemplatePageViewModel(creationInfo, _dataObjectServiceMock.Object);
            
            //when
            viewModel.Load();

            //then
            _dataObjectServiceMock.Verify(d => d.GetElementStructureAsync(SystemObjectIds.RootObjectId, It.IsAny<Action>(), false), Times.Once);
        }

        [Test]
        public void should_can_go_next_when_selected_project()
        {
            //Given
            var creationInfo = new CreationInfo();
            var type = TestTools.GetITypeMock();
            var parentType = TestTools.GetITypeMock();
            parentType.Setup(t => t.Children).Returns(new ReadOnlyCollection<int>(new List<int>() {type.Object.Id}));
            var parent = TestTools.GetIDataObjectMock(parentType);
            creationInfo.Parent = parent.Object;

            var viewModel = new SelectProjectTemplatePageViewModel(creationInfo, _dataObjectServiceMock.Object);
            
            //Проект монтируется на диск
            type.Setup(t => t.IsMountable).Returns(true);
            var idataObjectMock = TestTools.GetIDataObjectMock(type);

            //when
            var node = TestTools.GetNodeViewModel(idataObjectMock, _dataObjectServiceMock);
            viewModel.ObjectsTreeViewModel.SelectedItem = node;
            
            //Then
            Assert.IsTrue(viewModel.CanGoNext());
        }

        [Test]
        public void should_not_can_go_next_when_selected_project()
        {
            //Given
            var creationInfo = new CreationInfo();
            var viewModel = new SelectProjectTemplatePageViewModel(creationInfo, _dataObjectServiceMock.Object);

            var type = TestTools.GetITypeMock();
            //Элемент не монтируется на диск
            type.Setup(t => t.IsMountable).Returns(false);
            var idataObjectMock = TestTools.GetIDataObjectMock(type);
            creationInfo.Parent = TestTools.GetIDataObjectMock(idataObjectMock.Object.ParentId).Object;

            //when
            var node = TestTools.GetNodeViewModel(idataObjectMock, _dataObjectServiceMock);
            viewModel.ObjectsTreeViewModel.SelectedItem = node;

            //Then
            Assert.IsFalse(viewModel.CanGoNext());

            //Снимем селек в дереве
            viewModel.ObjectsTreeViewModel.SelectedItem = null;
            //Then
            Assert.IsFalse(viewModel.CanGoNext());
        }

        [Test]
        public void should_save_selected_project()
        {
            //Given
            var creationInfo = new CreationInfo();
            var viewModel = new SelectProjectTemplatePageViewModel(creationInfo, _dataObjectServiceMock.Object);

            var type = TestTools.GetITypeMock();
            //Элемент не монтируется на диск
            type.Setup(t => t.IsMountable).Returns(true);
            var idataObjectMock = TestTools.GetIDataObjectMock(type);

            //when
            var selectedNode = TestTools.GetNodeViewModel(idataObjectMock, _dataObjectServiceMock);
            viewModel.ObjectsTreeViewModel.SelectedItem = selectedNode;

            //then
            Assert.AreEqual(creationInfo.Template, selectedNode.Source);
        }

        [Test]
        public void should_update_state_when_select()
        {
            //Given
            var creationInfo = new CreationInfo();
            var viewModel = new SelectProjectTemplatePageViewModel(creationInfo, _dataObjectServiceMock.Object);

            var type = TestTools.GetITypeMock();
            var idataObjectMock = TestTools.GetIDataObjectMock(type);
            var selectedNode = TestTools.GetNodeViewModel(idataObjectMock, _dataObjectServiceMock);

            var  isUpdateEventRaised = false;
            viewModel.UpdateState += (sender, args) =>
            {
                isUpdateEventRaised = true;
            };
            
            //when
            viewModel.ObjectsTreeViewModel.SelectedItem = selectedNode;

            //then
            Assert.IsTrue(isUpdateEventRaised);
        }
    }
}
