/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Windows;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Controls.ObjectCardView;
using Ascon.Pilot.SDK.Extensions;
using Ascon.Pilot.Theme.Controls;
using DynamicData;
using ProjectCloneWizard.Dialogs;
using ProjectCloneWizard.ObjectsTree;
using ProjectCloneWizard.ViewModels;
using IDataObject = Ascon.Pilot.SDK.IDataObject;

namespace ProjectCloneWizard.Common
{
    public interface ICopyStructureService
    {
        Task<Guid> CopyStructure(IDataObject rootElement, IDataObject parent = null);
        void SetElementToCopy(IDataObject selected);
        IDataObject GetElementToCopy();
    }

    [Export(typeof(ICopyStructureService))]
    class CopyStructureService : ICopyStructureService
    {
        private readonly IObjectsRepository _repository;
        private readonly IObjectModifier _modifier;
        private readonly IFileProvider _fileProvider;
        private IDataObject _elementToCopy;

        [ImportingConstructor]
        public CopyStructureService(IObjectsRepository repository, IObjectModifier modifier, IFileProvider fileProvider)
        {
            _repository = repository;
            _modifier = modifier;
            _fileProvider = fileProvider;
        }

        public async Task<Guid> CopyStructure(IDataObject rootElement, IDataObject parent = null)
        {
            var dataObjectServiceFactory = new AllDataObjectServiceFactory(_repository);
            var service = dataObjectServiceFactory.GetService();
            var elements = await service.GetElementStructureAsync(rootElement);
            var nodesToCopy = elements.Select(o => new NodeViewModel(new Node<IDataObject, Guid>(o, o.Id), service, false)).ToList();

            if (parent == null)
                parent = await _repository.GetObjectAsync(rootElement.ParentId, CancellationToken.None);

            var attributesToCopy = new Dictionary<string, DValue>();
            foreach (var attribute in rootElement.Attributes)
            {
                attributesToCopy[attribute.Key] = DValue.GetDValue(attribute.Value);
            }

            var creationInfo = new CreationInfo
            {
                CopyAccessForObjects = true,
                Template = rootElement,
                ObjectsNodes = nodesToCopy,
                Attributes = attributesToCopy,
                Parent = parent
            };

            var progress = new ProcessPageViewModel(creationInfo, _modifier, _repository, _fileProvider);
            if (nodesToCopy.Count > 30)
            {
                var dialogView = new DialogView();
                var dialogViewModel = new DialogViewModel(progress);
                dialogView.DataContext = dialogViewModel;

                var dialog = new DialogWindow
                {
                    Content = dialogView,
                    Title = $"Копирование...",
                    Padding = new Thickness(0, 0, 0, 0),
                    WindowStartupLocation = WindowStartupLocation.CenterScreen,
                    Width = 500,
                    Height = 200,
                    ShowInTaskbar = true,
                    ResizeMode = ResizeMode.NoResize,
                    Uid = "CopyElementWindow",
                    SizeToContent = SizeToContent.Height,
                };

                _ = progress.Start().ContinueWith((task) => dialog.Close(), TaskScheduler.FromCurrentSynchronizationContext());
                dialog.ShowDialog();
            }
            else
            {
                await progress.Start();
            }

            return creationInfo.ObjectIdToShow;
        }

        public void SetElementToCopy(IDataObject selected)
        {
            _elementToCopy = selected;
        }

        public IDataObject GetElementToCopy()
        {
            return _elementToCopy;
        }
    }
}
