/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Linq;
using Ascon.Pilot.DataModifier;
using Pilot.Xps.Domain;
using Pilot.Xps.Domain.CertificateStorage;
using Pilot.Xps.Domain.Signatures;
using Pilot.Xps.Entities;

namespace Pilot.Xps.Connection.Server
{
    class XpsServiceApi : IXpsServiceApi
    {
        private readonly IXpsDocumentMergeService _mergeService;
        private readonly IXpsDocumentSigner _documentSigner;
        private readonly IBackend _backend;
        private readonly ICertificateStorage _certificateStorage;

        public XpsServiceApi(IXpsDocumentMergeService mergeService, IXpsDocumentSigner documentSigner, IBackend backend, ICertificateStorage certificateStorage)
        {
            _mergeService = mergeService;
            _documentSigner = documentSigner;
            _backend = backend;
            _certificateStorage = certificateStorage;
        }

        public bool CheckConnected()
        {
            // метод для проверки коннекта
            return true;
        }

        public byte[] MergeXpsDocument(Guid documentId)
        {
            var document = _backend.GetObject(documentId);
            if (document == null)
                return null;

            var type = _backend.GetType(document.TypeId);;
            return _mergeService.MergeDocument(document, type);
        }

        public SignResult SignDocument(Guid documentId, int personId)
        {
            var document = _backend.GetObject(documentId);
            if (document == null)
                throw new Exception("Document not found");

            var person = _backend.GetPerson(personId);
            if (person == null)
                throw new Exception("Person not found");

            var certificate = _certificateStorage.GetCertificate2(person);
            if (certificate == null)
                throw new Exception("Certificate not found");

            _documentSigner.SignDocumentWithCertificate(document, person.Positions, certificate);
            return SignResult.SignedSuccessfully;
        }

        public SignResult SignDocument(Guid documentId, int[] positions, int personId)
        {
            var document = _backend.GetObject(documentId);
            if (document == null)
                throw new Exception("Document not found");

            var person = _backend.GetPerson(personId);
            if (person == null)
                throw new Exception("Person not found");

            var certificate = _certificateStorage.GetCertificate2(person);
            if (certificate == null)
                throw new Exception("Certificate not found");

            _documentSigner.SignDocumentWithCertificate(document, positions, certificate);
            return SignResult.SignedSuccessfully;
        }

        public byte[] GetSignatures(Guid documentId)
        {
            var document = _backend.GetObject(documentId);
            if (document == null)
                throw new Exception("Document not found");

            var signatures = _documentSigner.GetSignatures(document);
            var buffer = XpsDigitalSignatureSerializer.Serialize(signatures);
            return buffer;
        }

        public byte[] GetSignatures(Guid documentId, DateTime snapshotDateTime)
        {
            var document = _backend.GetObject(documentId);
            if (document == null)
                throw new Exception("Document not found");

            var signatures = _documentSigner.GetSignatures(document, snapshotDateTime);
            var buffer = XpsDigitalSignatureSerializer.Serialize(signatures);
            return buffer;
        }
    }
}
