/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using Ascon.Pilot.SDK;
using Ascon.Pilot.Theme.Controls;
using Ascon.Pilot.Theme.Tools;
using PilotShare.Client.Properties;
using PilotShare.Client.View;
using IDataObject = Ascon.Pilot.SDK.IDataObject;

namespace PilotShare.Client
{
    public interface IPilotShareDialogService
    {
        void ShowExternalLinkManager(IDataObject document);
        bool? ShowEditLinkDialog(LinkItem link);
        bool? ShowCreateLinkDialog(IDataObject target, LinkItem link);
    }

    [Export(typeof(IPilotShareDialogService))]
    public class DialogService : IPilotShareDialogService
    {
        private readonly IObjectsRepository _repository;
        private readonly ILinkObjectModifier _modifier;
        private readonly IObjectModifier _modifier1;
        private readonly IPersonalSettings _settings;
        private readonly ISearchService _searchService;
        private readonly ISettingsService _settingsService;

        [ImportingConstructor]
        public DialogService(
            IObjectsRepository repository,
            ILinkObjectModifier modifier,
            IPersonalSettings settings,
            IObjectModifier modifier1,
            ISettingsService settingsService,
            ISearchService searchService)
        {
            _repository = repository;
            _modifier = modifier;
            _modifier1 = modifier1;
            _settings = settings;
            _settingsService = settingsService;
            _searchService = searchService;
        }


        public void ShowExternalLinkManager(IDataObject document)
        {
            using (var vm = new LinksManagerViewModel(document, _searchService, _repository, _modifier, _settingsService, this))
            {
                var view = new LinksManagerView { DataContext = vm };
                var dialog = new PureWindow
                {
                    Title = $"{Resources.LinksManagerTitle} - \"{document.DisplayName}\"",
                    Content = view,
                    Width = 900,
                    Height = 500,
                    DataContext = vm,
                    ResizeMode = ResizeMode.CanResize,
                    WindowStartupLocation = WindowStartupLocation.CenterScreen
                };

                dialog.Show();
            }
        }

        public bool? ShowEditLinkDialog(LinkItem link)
        {
            var view = new EditLinkView();
            var vm = new EditLinkViewModel(link, _modifier, view);
            view.DataContext = vm;
            var dialog = new DialogWindow
            {
                Title = Resources.EditExternalLinkTitle,
                Content = view,
                Width = 500,
                Height = 400,
                SizeToContent = SizeToContent.Width,
                DataContext = vm,
                ResizeMode = ResizeMode.NoResize,
                WindowStartupLocation = WindowStartupLocation.CenterScreen
            };

            dialog.SetParentWindow(LayoutHelper.GetOwnerWindow());
            return dialog.ShowDialog();
        }

        public bool? ShowCreateLinkDialog(IDataObject target, LinkItem link)
        {
            var view = new EditLinkView();
            var vm = new CreateLinkViewModel(target, link, _modifier, view, _repository);
            view.DataContext = vm;
            var dialog = new DialogWindow
            {
                Title = Resources.NewExternalLink,
                Content = view,
                Width = 500,
                Height = 400,
                SizeToContent = SizeToContent.Width,
                DataContext = vm,
                ResizeMode = ResizeMode.NoResize,
                WindowStartupLocation = WindowStartupLocation.CenterScreen
            };

            dialog.SetParentWindow(LayoutHelper.GetOwnerWindow());
            return dialog.ShowDialog();
        }
    }
}
