/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.ServiceModel.Security;
using System.Threading.Tasks;
using PilotShare.Core;

namespace PilotShare.Server.Models
{
    public interface IFileLoaderService
    {
        Task<byte[]> LoadFilesAsync(Guid linkId, string password);
    }

    class FileLoaderService : IFileLoaderService
    {
        private readonly ILinkStateProvider _stateProvider;
        private readonly IFileLoader _fileLoader;

        public FileLoaderService(ILinkStateProvider stateProvider, IFileLoader fileLoader)
        {
            _stateProvider = stateProvider;
            _fileLoader = fileLoader;
        }

        public async Task<byte[]> LoadFilesAsync(Guid linkId, string password)
        {
            var definition = await _stateProvider.GetLinkDefinitionAsync(linkId);
            if (!string.IsNullOrEmpty(definition.Password) && definition.Password != password)
            {
                throw new SecurityAccessDeniedException("Password is incorrect");
            }

            return await DownloadFilesAsync(definition);
        }

        private Task<byte[]> DownloadFilesAsync(LinkDefinition definition)
        {
            return Task<byte[]>.Factory.StartNew(() =>
            {
                using (var compressedFileStream = new MemoryStream())
                { 
                    using (var zipArchive = new ZipArchive(compressedFileStream, ZipArchiveMode.Update, true))
                    {
                        foreach (var entry in GetFileEntries(new[] { definition.Item }))
                        {
                            WriteEntry(zipArchive, string.Empty, entry);
                        }
                    }

                    var data = compressedFileStream.ToArray();
                    return data;
                }
            });
        }

        private void WriteEntry(ZipArchive zipArhive, string root, Item item)
        {
            if (!string.IsNullOrEmpty(item.FileId))
            {
                var zipEntryName = Path.Combine(root, item.Name);
                var zipEntry = zipArhive.CreateEntry(zipEntryName, CompressionLevel.NoCompression);
                
                var fileId = Guid.Parse(item.FileId);
                var fileBody = _fileLoader.Download(fileId, item.FileSize);

                using (var originalFileStream = new MemoryStream(fileBody))
                using (var zipEntryStream = zipEntry.Open())
                {
                    originalFileStream.CopyTo(zipEntryStream);
                }
            }
        }

        IEnumerable<Item> GetFileEntries(IEnumerable<Item> items)
        {
            foreach (var item in items)
            {
                if (!string.IsNullOrEmpty(item.FileId))
                    yield return item;

                foreach (var child in GetFileEntries(item.Children))
                {
                    yield return child;
                }
            }
        }
    }
}
