/*
Copyright © 2024 ASCON-Design Systems LLC. All rights reserved.
This sample is licensed under the MIT License.
*/


// @ts-ignore
import { ASN1 } from "@lapo/asn1js";
// @ts-ignore
import { Defs } from '@lapo/asn1js/defs.js';

declare type DefsCommonTypes = [ string [] ];

interface AsnTypeDef {
  content: AsnTypeDef [];
  id?: string
  class?: string;
  name?: string;
  type?: string;
  explicit?: boolean;
}

interface AsnModuleDef {
  name: string;
  oid: string;
  source: string;
}

interface AsnDef {
  name: string;
  type: AsnTypeDef;
  id?: string;
  description?: string;
  module?: AsnModuleDef;
  mismatch?: number;
}

type AsnWithoutSubObject = Omit<ASN1, 'sub'>;
export type AsnObject = AsnWithoutSubObject & {sub: AsnObject[] | null, def: AsnDef };

export type IAllowedSearchingPropNames = 'id' | 'type' | 'name';


export class Asn1Parser {

  static searchInTree(node: AsnObject, id: string, propName: IAllowedSearchingPropNames = 'id'): AsnObject | null {
    if (node && node.def[propName] === id) {
      return node;
    }
    if (node && node.sub)
      for (let i = 0; i < node.sub.length; i++) {
        const child = this.searchInTree(node.sub[i], id, propName);
        if (child) {
          return child;
        }
      }
    return null;
  }

  static searchInBlock(node: AsnObject, type: string, value: string, nodeParent: AsnObject, propName: IAllowedSearchingPropNames = 'id'): AsnObject | null {
    const content: string = node?.content();

    if (!content) {
      return null;
    }

    const oidName = content.split(/\n/).filter(token => token === value);

    if (node && node.def[propName] === type && oidName.length > 0) {
      return nodeParent;
    }

    if (node && node.sub)
      for (let i = 0; i < node.sub.length; i++) {
        nodeParent = node;
        const child = this.searchInBlock(node.sub[i], type, value, nodeParent, propName);
        if (child)
          return child;
      }
    return null;
  }

  static searchByTag(node: AsnObject, tagNumber: number = -1, tagConstructed: boolean = false, nodeParent: AsnObject | null = null): AsnObject | null {
    if (tagNumber === -1) return null;

    if (node && node.tag.tagNumber === tagNumber && node.tag.tagConstructed === tagConstructed)
      return nodeParent ? nodeParent : node;

    if (node && node.sub)
      for (let i = 0; i < node.sub.length; i++) {
        if (nodeParent)
          nodeParent = node;
        const child = this.searchByTag(node.sub[i], tagNumber, tagConstructed, nodeParent);
        if (child)
          return child;
      }
    return null;
  }

  static getSubjectKeyIdentifier(node: AsnObject): string | null {
    const extensionValue = this.getExtensionValue(node, 'subjectKeyIdentifier');
    return extensionValue?.sub?.[0]?.content()?.split('\n')?.[1];
  }

  static getAuthorityKeyIdentifier(node: AsnObject): string | null {
    const extensionValue = this.getExtensionValue(node, 'authorityKeyIdentifier');
    return extensionValue?.sub?.[0]?.sub?.[0]?.content()?.split('\n')?.[1];
  }

  static initAsn(sign: Uint8Array): AsnObject {
    const asnObject = ASN1.decode(sign); 
    const types = (Defs.commonTypes as DefsCommonTypes).map((type) => {
      const stats = Defs.match(asnObject, type);
      return { type, match: stats.recognized / stats.total };
    }).sort((a, b) => b.match - a.match);

    Defs.match(asnObject, types[0].type);
    return asnObject as AsnObject;
  }
 
  private static getExtensionValue(node: AsnObject, name: string): AsnObject | null {
    if (!name)
      return null;

    const extensions = Asn1Parser.searchInTree(node, 'extensions');

    for (const extensionsList of extensions?.sub || []) {
      for (const extension of extensionsList?.sub || []) {
        const extensionId = Asn1Parser.searchInTree(extension, 'extnID');
        const value = extensionId?.content()?.split('\n')?.[1];

        if (value === name) {
          const extensionValue = Asn1Parser.searchInTree(extension, 'extnValue');
          return extensionValue;
        }
      }
    }

    return null;
  }
}