/*
Copyright © 2024 ASCON-Design Systems LLC. All rights reserved.
This sample is licensed under the MIT License.
*/

import { AlgorithmDetector } from './algorithm.detector';

export class CryptoProHashCalculator {

  progress!: (persent: number) => void;

  async calculateHash(buffer: ArrayBuffer, signOrCertificate: ArrayBuffer | ICryptoProCertificate): Promise<ICryptoProHashedData> {
    return new Promise(async (resolve, reject) => {
      try {
        const file = new File([buffer], "");
        const chunkSize = 3 * 1024 * 1024; // 3MB
        const chunks = Math.ceil(buffer.byteLength / chunkSize);
        let currentChunk = 0;
        
        const oHashedData = <ICryptoProHashedData> await cadesplugin.CreateObjectAsync('CAdESCOM.HashedData');
        await oHashedData.propset_DataEncoding(cadesplugin.CADESCOM_BASE64_TO_BINARY);
        const hashAlg = await AlgorithmDetector.detectDigestAlgFrom(signOrCertificate);
        await oHashedData.propset_Algorithm(hashAlg);      

        const onFileLoaded = async (ev: ProgressEvent<FileReader>) => {
          const sFileData = ev?.target?.result as string;
          const sBase64Data = this.removeBase64Prefix(sFileData);

          await oHashedData.Hash(sBase64Data);

          currentChunk++;

          const percentLoaded = Math.round((currentChunk / chunks) * 100);
          if (percentLoaded <= 100 && this.progress)
            this.progress(percentLoaded);

          if (currentChunk < chunks)
            loadNextChunk();
          else {
            resolve(oHashedData);
          }
        }

        const onFileReadingError = () => {
          reject("File reading error");
        }

        const loadNextChunk = () => {
          const fileReader = new FileReader();
          fileReader.onload = onFileLoaded;
          fileReader.onerror = onFileReadingError;
          const start = currentChunk * chunkSize;
          const end = start + chunkSize >= file.size ? file.size : start + chunkSize;
          fileReader.readAsDataURL(file.slice(start, end));
        }

        loadNextChunk();
      } catch (error) {
        reject(error);
      }
    });
  }

  private removeBase64Prefix(base64String: string): string {
    const header = ";base64,";
    const startIndex = base64String.indexOf(header);
    if (startIndex !== -1) {
        return base64String.substring(startIndex + header.length);
    }
    return base64String;
  }
}