/*
Copyright © 2024 ASCON-Design Systems LLC. All rights reserved.
This sample is licensed under the MIT License.
*/
import { firstValueFrom, map, Observable, ReplaySubject, takeUntil } from "rxjs";
import { ICertificate, ICryptoProvider, 
         ISignatureVerificationResult, ISignatureRequest, 
         IFile,
         IImportedSignatureVerificationResult} from "@pilotdev/pilot-web-sdk";
import { FileSignatureUpdater } from "./file-signature.updater";
import { Queue, QueueType } from "./utils/queue";
import { IAdapter } from "./cryptopro/adapter";
import { signatureAlgorithms } from "./supportedAlgorithms";
import { IAdvancementSettingsProvider } from "./advancement/advancementSettingsProvider";
import { SignatureParser } from "./cryptopro/signature.parser";
import { convertToString } from "./utils/utils";

export class CryptoProExtensionModel implements ICryptoProvider {
  private _destroy$ = new ReplaySubject<void>(1);

  constructor(
    private readonly  _cryptoProAdapter: IAdapter,
    private readonly _fileSignatureUpdater: FileSignatureUpdater,
    private readonly _queue: Queue<QueueType>,
    private readonly _enhancementSettingsProvider: IAdvancementSettingsProvider
  ) {
  }
  
  initialize(): void {
    this._enhancementSettingsProvider.initialize(this._destroy$).pipe(takeUntil(this._destroy$)).subscribe(async () => {
      await this._cryptoProAdapter.init();
    });
  }

  dispose(): void {
    this._destroy$.next();
    this._destroy$.complete();
    this._queue.stop();
  }

  sign(documentId: string, actualFile: IFile, arrayBuffer: ArrayBuffer, certificate: ICertificate, signatureRequestIds: string[]): Observable<string> {
    const item = this._queue.enqueue(() => this.signAction(documentId, actualFile, arrayBuffer, certificate, signatureRequestIds)) as Observable<string>;
    return item;
  }

  verify(file: ArrayBuffer, sign: ArrayBuffer, signatureRequset: ISignatureRequest): Observable<ISignatureVerificationResult> {
    return this._queue.enqueue(() => this._cryptoProAdapter.verify(file, sign, signatureRequset)) as Observable<ISignatureVerificationResult>;
  }

  verifyImportedSignature(file: ArrayBuffer, sign: ArrayBuffer): Observable<IImportedSignatureVerificationResult> {
    return this._queue.enqueue(() => this._cryptoProAdapter.verifyImportedSignature(file, sign)) as Observable<IImportedSignatureVerificationResult>;
  }
  
  getCertificates(): Observable<ICertificate[]> {
    return this._queue.enqueue(() => this.getCertificatesAction()) as Observable<ICertificate[]>;
  }
  
  canProcessAlgorithms(publicKeyOid: string): boolean {
    return signatureAlgorithms.has(publicKeyOid);
  }

  canProcessSignature(signatureFile: ArrayBuffer): boolean {
    try {
      const signBase64 = convertToString(signatureFile);
      const parsedSig = SignatureParser.parse(signBase64);
      return signatureAlgorithms.has(parsedSig.signatureAlgorithm);
    } catch (err) {
      return false;
    }
  }

  onCryptoProPluginLoaded(error?: Error): void {
    this._queue.start();
    if (error) {
      console.error(error);
      return;
    }

    console.debug("CryptoPro plugin has been loaded successfully");
  }

  private async signAction(documentId: string, actualFile: IFile, arrayBuffer: ArrayBuffer, certificate: ICertificate, signatureRequestIds: string[]): Promise<string> {
    const signBase64 = await this._cryptoProAdapter.sign(arrayBuffer, certificate);
    return firstValueFrom(
      this._fileSignatureUpdater
        .setSignToObjectFile(documentId, actualFile, signBase64, certificate.publicKeyOid, signatureRequestIds, certificate)
        .pipe(map((v) => signBase64))
    );
  }

  private async getCertificatesAction(): Promise<ICertificate[]> {
    return this._cryptoProAdapter.getCertificates();
  }
}