﻿/*
  Copyright © 2025 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using Newtonsoft.Json;
using PdfStamper.Settings;
using Pilot.WebServer.SDK;
using System.Text;

namespace PdfStamper;

internal class FileStampProvider
{
    private readonly IFileLoader _fileLoader;
    private readonly IModifierProvider _modifierProvider;

    public FileStampProvider(IFileLoader fileLoader, IModifierProvider modifierProvider)
    {
        _fileLoader = fileLoader;
        _modifierProvider = modifierProvider;
    }

    public StampPosition? GetStampPosition(PdfDocumentContext context, string stampId)
    {
        var snapshot = Utils.GetFilesSnapshot(context);
        if (snapshot == null)
            return null;

        var currentFile = snapshot.Files.FirstOrDefault(n => n.Name.Equals($"{stampId}{Constants.SIGNATURE_STAMP_EXT}"));
        if (currentFile == null) 
            return null;

        var downloaded = _fileLoader.Download(currentFile.Id, currentFile.Size);
        return DeserializeStampPosition(downloaded);
    }

    public void UpdateStampPosition(StampPositionArgs args)
    {
        var snapshot = Utils.GetFilesSnapshot(args.Context);
        if (snapshot == null)
            return;

        var stampsFilename = $"{args.StampId}{Constants.SIGNATURE_STAMP_EXT}";
        var currentFile = snapshot.Files.FirstOrDefault(n => n.Name.Equals(stampsFilename));
        var modifier = _modifierProvider.NewModifier();
        var builder = modifier.Edit(Guid.Parse(args.Context.Document.Id));
        var stream = GetStreamFromStampData(new StampPosition { X = args.X, Y = args.Y });

        if (currentFile != null)
            builder.ReplaceFile(currentFile.Name, stream, currentFile.Id, currentFile.Created, DateTime.Now, DateTime.Now, snapshot.Created);
        else
            builder.AddFile(stampsFilename, stream, DateTime.Now, DateTime.Now, DateTime.Now, args.Context.Version, out _);

        modifier.Apply();
    }

    private string SerializeStampPosition(StampPosition position)
    {
        return JsonConvert.SerializeObject(position, new JsonSerializerSettings { Formatting = Formatting.Indented });
    }

    private StampPosition? DeserializeStampPosition(byte[] jsonBytes)
    {
        var json = Encoding.UTF8.GetString(jsonBytes);
        if (json == null)
            return null;

        return JsonConvert.DeserializeObject<StampPosition>(json);
    }

    private Stream GetStreamFromStampData(StampPosition stampPosition)
    {
        var newDataString = SerializeStampPosition(stampPosition);
        var data = Encoding.UTF8.GetBytes(newDataString);
        var stream = new MemoryStream();
        stream.Write(data, 0, data.Length);
        return stream;
    }
}
