import { AsyncPipe } from '@angular/common';
import { AfterViewInit, Component, ElementRef, inject, OnInit, Renderer2, ViewChild, ViewEncapsulation } from '@angular/core';
import { MatButtonModule } from '@angular/material/button';
import { MatTableModule } from '@angular/material/table';
import { MatTooltip } from '@angular/material/tooltip';
import { BimRightPanelContext, IBimFeatures, IBimModelSearchResult, IBimSearchSetData, IThemeService, Theme } from '@pilotdev/pilot-web-sdk';
import { BehaviorSubject, map, Observable } from 'rxjs';
import { BIM_FEATURES, THEME_SERVICE } from '../../tokens';

@Component({
  selector: 'app-right-panel-view',
  standalone: true,
  imports: [
    MatButtonModule,
    MatTableModule,
    MatTooltip,
    AsyncPipe
  ],
  templateUrl: './right-panel-view.component.html',
  styleUrl: './right-panel-view.component.scss',
  encapsulation: ViewEncapsulation.ShadowDom
})
export class RightPanelViewComponent implements OnInit, AfterViewInit {
  private readonly _lastExecuteResults$: BehaviorSubject<IBimModelSearchResult[]> = new BehaviorSubject<IBimModelSearchResult[]>([]);

  private readonly _themeService: IThemeService;
  private readonly _renderer: Renderer2;
  private readonly _bimFeatures: IBimFeatures;
  private readonly _context: BimRightPanelContext;

  @ViewChild('wrapperElement') wrapperDiv!: ElementRef<HTMLDivElement>;

  searchSets$!: Observable<IBimSearchSetData[]>;

  constructor() {
    this._themeService = inject(THEME_SERVICE);
    this._renderer = inject(Renderer2);

    this._bimFeatures = inject(BIM_FEATURES);
    this._context = inject(BimRightPanelContext);
  }

  get lastExecuteResults$(): Observable<IBimModelSearchResult[]> {
    return this._lastExecuteResults$.asObservable();
  }

  ngOnInit(): void {
    this.loadSearchSets();
    this.subscribeResultChanges();
  }

  ngAfterViewInit(): void {
    this.subscribeThemeChanges();
  }

  onSearchSetExecute(data: IBimSearchSetData): void {
    const modelPartIds = this._context.viewer.model
      .getAllModelParts()
      .map((mp) => mp.id);
    
    this._bimFeatures.searchSetService
      .executeSearchSet(data.id, modelPartIds)
      .subscribe((res) => this._lastExecuteResults$.next(res));
  }

  onResultsClear(): void {
    this._lastExecuteResults$.next([]);
  }

  private loadSearchSets(): void {
    this.searchSets$ = this._bimFeatures.searchSetService
      .getRecursivelyFromModel(this._context.modelId)
      .pipe(map(sets => sets.filter(s => !s.isDeleted && !s.isFolder)));
  }

  private subscribeResultChanges(): void {
    const model = this._context.viewer.model;

    this._lastExecuteResults$.subscribe(results => {
      model.clearColors();

      results.forEach(r => model.setColor(r.globalIdReadable, 255, 0, 0, 1, r.modelPartId));
    })
  }

  private subscribeThemeChanges(): void {
    this._themeService.change.subscribe((theme) => {
      if (theme.theme === Theme.Dark) {
        this._renderer.addClass(this.wrapperDiv.nativeElement, 'theme-dark');
      } else {
        this._renderer.removeClass(this.wrapperDiv.nativeElement, 'theme-dark');
      }
    });
  }
}
