/**
 * Copyright (c) 2024 ASCON-Design Systems LLC. All rights reserved.
 * This sample is licensed under the MIT License.
 */
import {
  Guid,
  IDataObject,
  IDisposable,
  IInitializable,
  IModifierProvider,
  InjectionSource,
  IOpenspaceView,
  ObjectsViewContext,
} from '@pilotdev/pilot-web-sdk';
import { StrictMode, createElement } from 'react';
import { createRoot, Root } from 'react-dom/client';
import { FileTableView, FileTableViewProps } from '../../components/FileTableView';
import { FileTabId } from '../documentTabs';

export class FileTableViewSample implements IOpenspaceView<ObjectsViewContext>, IDisposable, IInitializable {
  private _rootElement: HTMLElement | undefined;
  private _reactRoot: Root | undefined;
  private _selectedObject: IDataObject | undefined;
  private _modifierProvider: IModifierProvider | undefined;

  initialize(injectionSource: InjectionSource): void {
    this._modifierProvider = injectionSource.modifierProvider;
  }

  getViewId(): string {
    return FileTabId;
  }

  getView(context: ObjectsViewContext): HTMLElement | undefined {
    this._selectedObject = context.selectedObjects[0];

    if (!this._rootElement) {
      this._rootElement = document.createElement('div');
    }

    if (!this._reactRoot) {
      this._reactRoot = createRoot(this._rootElement);
    }

    const element = createElement(FileTableView, {
      files: context.selectedObjects[0].actualFileSnapshot.files ?? [],
      onFileAdded: this.onFileAdded.bind(this),
      onDeleteAllFiles: this.onDeleteAllFiles.bind(this),
    } as unknown as FileTableViewProps);

    const strictModeElement = createElement(StrictMode, { children: element });

    if (this._reactRoot) {
      this._reactRoot.render(strictModeElement);
    }

    return this._rootElement;
  }

  dispose(): void {
    if (this._reactRoot) {
      this._reactRoot.unmount();
      this._reactRoot = undefined;
    }

    if (this._rootElement) {
      this._rootElement = undefined;
    }
  }

  private onFileAdded(file: File | undefined): void {
    if (!this._modifierProvider || !this._selectedObject || !file) 
      return;

    const modifier = this._modifierProvider.newModifier();
    const builder = modifier.edit(this._selectedObject.id);
    const created = new Date();
    builder.addFile(
      Guid.newGuid(),
      file,
      created,
      created,
      created
    );
    modifier.apply().subscribe();
  }

  private onDeleteAllFiles(): void {
    if (!this._modifierProvider || !this._selectedObject) 
      return;

    const modifier = this._modifierProvider.newModifier();
    const builder = modifier.edit(this._selectedObject.id);
    this._selectedObject.actualFileSnapshot.files.forEach(file => {
      builder.removeFile(file.body.id);
    });
    modifier.apply().subscribe();
  }
}
