﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Globalization;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Text;
using System.Windows.Input;
using System.Windows.Threading;
using Ascon.Pilot.Bim.SDK.ModelSamplesHelpers;
using Ascon.Pilot.Bim.SDK.ModelStorage;
using Ascon.Pilot.Bim.SDK.ModelStorageSample.Converters.Parameters;
using Ascon.Pilot.SDK;
using Prism.Commands;

namespace Ascon.Pilot.Bim.SDK.ModelStorageSample
{
    public class PluginViewModel : INotifyPropertyChanged
    {
        private readonly IPilotServiceProvider _pilotServiceProvider;
        private readonly Dispatcher _dispatcher;
        private readonly IUpdateListener _updateListener;
        private IModelStorage _modelStorage;

        private DelegateCommand<string> _linkCommand;
        private DelegateCommand _getIsLoadedCommand;
        private DelegateCommand _getVersionsCommand;
        private DelegateCommand _getModelPartsIdsCommand;
        private DelegateCommand<string> _getRootElementsCommand;
        private DelegateCommand<CommandParameters> _getElementsCommand;
        private DelegateCommand<CommandParameters> _getElementCommand;
        private DelegateCommand<CommandParameters> _getChildElementsCommand;
        private DelegateCommand<CommandParameters> _getElementPropertiesCommand;
        private DelegateCommand<CommandParameters> _getElementMeshInstancesCommand;
        private DelegateCommand<string> _getMeshDefinitionCommand;
        private DelegateCommand _unlinkCommand;
        private DelegateCommand _clearLogCommand;


        public PluginViewModel(IPilotServiceProvider pilotServiceProvider, Dispatcher dispatcher, IUpdateListener updateListener)
        {
            _pilotServiceProvider = pilotServiceProvider;
            _dispatcher = dispatcher;
            _updateListener = updateListener;

            Log.CollectionChanged += (sender, args) =>
            {
                if (args.Action != NotifyCollectionChangedAction.Add)
                    return;

                _dispatcher.BeginInvoke((Action)(() =>
                {
                    _updateListener.OnLogUpdated();
                }));
            };

            InitCommands();
        }

        private void InitCommands()
        {
            _linkCommand = new DelegateCommand<string>(Link, x => !IsLinked);
            _getIsLoadedCommand = new DelegateCommand(GetIsLoaded, () => IsLinked);
            _getVersionsCommand = new DelegateCommand(GetVersions, () => IsLinked);
            _getModelPartsIdsCommand = new DelegateCommand(GetModelPartsIds, () => IsLinked);
            _getRootElementsCommand = new DelegateCommand<string>(GetRootElements, x => IsLinked);
            _getElementsCommand = new DelegateCommand<CommandParameters>(GetElements, x => IsLinked);
            _getElementCommand = new DelegateCommand<CommandParameters>(GetElement, x => IsLinked);
            _getChildElementsCommand = new DelegateCommand<CommandParameters>(GetChildElements, x => IsLinked);
            _getElementPropertiesCommand = new DelegateCommand<CommandParameters>(GetElementProperties, x => IsLinked);
            _getElementMeshInstancesCommand = new DelegateCommand<CommandParameters>(GetElementMeshInstances, x => IsLinked);
            _getMeshDefinitionCommand = new DelegateCommand<string>(GetMeshDefinition, x => IsLinked);
            _unlinkCommand = new DelegateCommand(Unlink, () => IsLinked);
            _clearLogCommand = new DelegateCommand(ClearLog);
        }

        public ObservableCollection<LogEntry> Log { get; } = new ObservableCollection<LogEntry>();

        public ICommand LinkCommand => _linkCommand;
        public ICommand GetIsLoadedCommand => _getIsLoadedCommand;
        public ICommand GetVersionsCommand => _getVersionsCommand;
        public ICommand GetModelPartsIdsCommand => _getModelPartsIdsCommand;
        public ICommand GetRootElementsCommand => _getRootElementsCommand;
        public ICommand GetElementsCommand => _getElementsCommand;
        public ICommand GetElementCommand => _getElementCommand;
        public ICommand GetChildElementsCommand => _getChildElementsCommand;
        public ICommand GetElementPropertiesCommand => _getElementPropertiesCommand;
        public ICommand GetElementMeshInstancesCommand => _getElementMeshInstancesCommand;
        public ICommand GetMeshDefinitionCommand => _getMeshDefinitionCommand;
        public ICommand UnlinkCommand => _unlinkCommand;
        public ICommand ClearLogCommand => _clearLogCommand;

        private bool _isLinked;
        public bool IsLinked
        {
            get => _isLinked;
            set
            {
                _isLinked = value;
                OnPropertyChanged(nameof(IsLinked));
                InvalidateCommands();
            }
        }

        private static DateTime? GetVersion(string version)
        {
            if (long.TryParse(version, out var result))
                return new DateTime(result);

            Messages.ShowErrorMessage("Version invalid");
            return null;
        }

        private static string GetElementsString(string header, List<IModelElement> elements, bool withCount = true)
        {
            var sb = new StringBuilder(elements.Any()
                ? (withCount ? $"{header} ({elements.Count}):\n" : $"{header}:\n")
                : $"No {header.ToLower()}");

            foreach (var element in elements)
                sb.Append($"Element id: {element.Id}, model part id: {element.Id.ModelPartId}, name: {element.Name}, type: {element.Type}\n");

            return sb.ToString();
        }

        private static string GetCoordinatesString(List<string> coordinates)
        {
            var sb = new StringBuilder();

            for (var i = 0; ;)
            {
                sb.Append($"({coordinates[i]},{coordinates[i + 1]},{coordinates[i + 2]})");

                i += 3;

                if (i < coordinates.Count && coordinates.Count - i >= 3)
                    sb.Append(",");
                else
                    break;
            }

            return sb.ToString();
        }

        public void OnPluginWindowClosed()
        {
            DisposeModelStorage();
        }

        private void Link(string modelId)
        {
            var modelStorage = GetModelStorage(modelId);

            if (modelStorage == null) 
                return;

            _modelStorage = modelStorage;
            _modelStorage.Loaded += OnLoaded;
            IsLinked = true;
            ClearLog();
            Log.Add(new LogEntry($"[{DateTime.Now.Ticks}] ===> Successfully linked to the model storage"));
        }

        private void GetIsLoaded()
        {
            Messages.ShowMessage($"Loaded: {_modelStorage.IsLoaded}", "Loading state");
        }

        private void GetVersions()
        {
            var versions = _modelStorage.GetVersions().OrderByDescending(v => v).ToList();

            var sb = new StringBuilder(versions.Any() ? $"Versions ({versions.Count}):\n" : "No versions");

            foreach (var version in versions)
                sb.Append($"{version.Ticks}\n");

            Messages.ShowMessage(sb.ToString(), "Versions");
        }

        private void GetModelPartsIds()
        {
            var ids = _modelStorage.GetModelPartsIds().ToList();

            var sb = new StringBuilder(ids.Any() ? $"Model parts ({ids.Count}):\n" : "No model parts");

            foreach (var id in ids)
                sb.Append($"{id}\n");

            Messages.ShowMessage(sb.ToString(), "Model parts ids");
        }

        private void GetRootElements(string version)
        {
            var modelVersion = GetVersion(version);

            if (modelVersion == null)
                return;

            var rootElements = _modelStorage.LoadRootElements(modelVersion.Value).ToList();
            const string header = "Root elements";
            var rootElementsString = GetElementsString(header, rootElements);

            Messages.ShowMessage(rootElementsString, header);
        }

        private void GetElements(CommandParameters parameters)
        {
            var id = Parsers.GetModelPartId(parameters.Ids);
            var version = GetVersion(parameters.Version);

            if (id == null || version == null)
                return;

            var elements = _modelStorage.LoadElements(id.Value, version.Value).ToList();
            const string header = "Elements";
            var elementsString = GetElementsString(header, elements);

            Messages.ShowMessage(elementsString, header);
        }

        private void GetElement(CommandParameters parameters)
        {
            var id = Parsers.GetModelElements(parameters.Ids).FirstOrDefault();
            var version = GetVersion(parameters.Version);

            if (id == null || version == null)
                return;

            var element = _modelStorage.LoadElement(id, version.Value);
            const string header = "Element";
            var elementsString = GetElementsString(header, element == null ? new List<IModelElement>() : new List<IModelElement> { element }, false);

            Messages.ShowMessage(elementsString, header);
        }

        private void GetChildElements(CommandParameters parameters)
        {
            var id = Parsers.GetModelElements(parameters.Ids).FirstOrDefault();
            var version = GetVersion(parameters.Version);

            if (id == null || version == null)
                return;

            var elements = _modelStorage.LoadChildElements(id, version.Value).ToList();
            const string header = "Child elements";
            var elementsString = GetElementsString(header, elements);

            Messages.ShowMessage(elementsString, header);
        }

        private void GetElementProperties(CommandParameters parameters)
        {
            var id = Parsers.GetModelElements(parameters.Ids).FirstOrDefault();
            var version = GetVersion(parameters.Version);

            if (id == null || version == null)
                return;

            var propertySets = _modelStorage.LoadElementProperties(id, version.Value).ToList();

            var sb = new StringBuilder(propertySets.Any()
                ? "Element properties:\n"
                : "No element properties");

            foreach (var propertySet in propertySets)
            {
                sb.Append($"\n● {propertySet.Name} ({propertySet.Type}):\n");
                foreach (var property in propertySet.Properties)
                {
                    sb.Append($" ► {property.Name}: {property.Value} {property.Unit}\n");
                }
            }

            Messages.ShowMessage(sb.ToString(), "Element properties");
        }

        private void GetElementMeshInstances(CommandParameters parameters)
        {
            var id = Parsers.GetModelElements(parameters.Ids).FirstOrDefault();
            var version = GetVersion(parameters.Version);

            if (id == null || version == null)
                return;

            var meshInstances = _modelStorage.LoadElementMeshInstances(id, version.Value).ToList();

            var sb = new StringBuilder(meshInstances.Any()
                ? "Element mesh instances:\n"
                : "No element mesh instances");

            foreach (var meshInstance in meshInstances)
            {
                sb.Append($"\nMesh id: {meshInstance.MeshDefinitionId.MeshId} \n");
                sb.Append($"Color: R = {meshInstance.Color.R}, G = {meshInstance.Color.G}, B = {meshInstance.Color.B}, A = {meshInstance.Color.A}\n");
                sb.Append("Placement:\n");
                sb.Append($"[{meshInstance.Placement.M11},\n");
                sb.Append($" {meshInstance.Placement.M12},\n");
                sb.Append($" {meshInstance.Placement.M13},\n");
                sb.Append($" {meshInstance.Placement.M14},\n");
                sb.Append($" {meshInstance.Placement.M21},\n");
                sb.Append($" {meshInstance.Placement.M22},\n");
                sb.Append($" {meshInstance.Placement.M23},\n");
                sb.Append($" {meshInstance.Placement.M24},\n");
                sb.Append($" {meshInstance.Placement.M31},\n");
                sb.Append($" {meshInstance.Placement.M32},\n");
                sb.Append($" {meshInstance.Placement.M33},\n");
                sb.Append($" {meshInstance.Placement.M34},\n");
                sb.Append($" {meshInstance.Placement.M41},\n");
                sb.Append($" {meshInstance.Placement.M42},\n");
                sb.Append($" {meshInstance.Placement.M43},\n");
                sb.Append($" {meshInstance.Placement.M44}]\n");
            }

            Messages.ShowMessage(sb.ToString(), "Element mesh instances");
        }

        private void GetMeshDefinition(string ids)
        {
            var id = Parsers.GetMeshDefinitions(ids).FirstOrDefault();

            if (id == null)
                return;

            var meshDefinition = _modelStorage.LoadMeshDefinition(new MeshDefinitionId(id.MeshId, id.ModelPartId));

            if (meshDefinition == null)
            {
                Messages.ShowMessage("No mesh definition", "Mesh definition");
                return;
            }

            var sb = new StringBuilder("Mesh definition:\n");

            sb.Append("Vertices:\n");
            var verticesString = GetCoordinatesString(meshDefinition.Vertices.Select(v => v.ToString(CultureInfo.InvariantCulture)).ToList());
            sb.Append($"{verticesString}\n");

            sb.Append("Normals:\n");
            var normalsString = GetCoordinatesString(meshDefinition.Normals.Select(n => n.ToString(CultureInfo.InvariantCulture)).ToList());
            sb.Append($"{normalsString}\n");

            sb.Append("Indices:\n");
            var indicesString = GetCoordinatesString(meshDefinition.Indices.Select(i => i.ToString(CultureInfo.InvariantCulture)).ToList());
            sb.Append($"{indicesString}\n");

            sb.Append("EdgeIndices:\n");
            var edgeIndicesString = GetCoordinatesString(meshDefinition.EdgeIndices.Select(e => e.ToString(CultureInfo.InvariantCulture)).ToList());
            sb.Append($"{edgeIndicesString}");

            Messages.ShowMessage(sb.ToString(), "Mesh definition");
        }

        private void Unlink()
        {
            DisposeModelStorage();
            IsLinked = false;
        }

        private void DisposeModelStorage()
        {
            if (_modelStorage == null)
                return;

            _modelStorage.Loaded -= OnLoaded;
            _modelStorage.Dispose();
            _modelStorage = null;
        }

        private void OnLoaded(object sender, EventArgs e)
        {
            _dispatcher.BeginInvoke((Action)(() =>
            {
                Log.Add(new LogEntry($"[{DateTime.Now.Ticks}] ===> Model storage is loaded"));
            }));
        }

        public void ClearLog()
        {
            Log.Clear();
        }

        private void InvalidateCommands()
        {
            _linkCommand.RaiseCanExecuteChanged();
            _getIsLoadedCommand.RaiseCanExecuteChanged();
            _getVersionsCommand.RaiseCanExecuteChanged();
            _getModelPartsIdsCommand.RaiseCanExecuteChanged();
            _getRootElementsCommand.RaiseCanExecuteChanged();
            _getElementsCommand.RaiseCanExecuteChanged();
            _getElementCommand.RaiseCanExecuteChanged();
            _getChildElementsCommand.RaiseCanExecuteChanged();
            _getElementPropertiesCommand.RaiseCanExecuteChanged();
            _getElementMeshInstancesCommand.RaiseCanExecuteChanged();
            _getMeshDefinitionCommand.RaiseCanExecuteChanged();
            _unlinkCommand.RaiseCanExecuteChanged();
        }

        private IModelStorage GetModelStorage(string modelId)
        {
            if (Guid.TryParse(modelId, out var modelIdValue))
            {
                var modelManager = _pilotServiceProvider.GetServices<IModelStorageProvider>().FirstOrDefault();
                var modelStorage = modelManager?.GetStorage(modelIdValue);

                if (modelStorage == null)
                    Messages.ShowErrorMessage("Can't link to the model storage");

                return modelStorage;
            }

            Messages.ShowErrorMessage("Model id invalid");
            return null;
        }

        public event PropertyChangedEventHandler PropertyChanged;

        protected virtual void OnPropertyChanged([CallerMemberName] string propertyName = null)
        {
            PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(propertyName));
        }
    }
}
