﻿/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Linq;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Windows;
using Ascon.Pilot.SDK;
using Ascon.Pilot.Bim.SDK.Model;
using Ascon.Pilot.Bim.SDK.ModelTab;
using Ascon.Pilot.Bim.SDK.ModelViewerSample.Tools;

namespace Ascon.Pilot.Bim.SDK.ModelViewerSample
{
    [Export(typeof(IDataPlugin))]
    public class ModelViewerSample : IDataPlugin
    {
        private readonly byte[] _icon;

        private readonly ITabsManager _tabsManager;
        private readonly IModelManager _modelManager;
        private readonly Dictionary<Guid, PluginViewModel> _tabViewModels = new Dictionary<Guid, PluginViewModel>();

        [ImportingConstructor]
        public ModelViewerSample(IPilotServiceProvider pilotServiceProvider, IObjectsRepository objectsRepository)
        {
            _tabsManager = pilotServiceProvider.GetServices<ITabsManager>().First();
            _modelManager = pilotServiceProvider.GetServices<IModelManager>().First();
            _icon = IconLoader.GetIcon("tab_icon.svg");
            var openedTabs = _tabsManager.GetTabs().OfType<IModelTab>().ToList();
            InitOpenedTabsSidebar(openedTabs);
            Subscribe();
        }

        private void InitOpenedTabsSidebar(List<IModelTab> openedTabs)
        {
            openedTabs.ForEach(AddSidebarTab);
        }

        private void AddSidebarTab(IModelTab modelTab)
        {
            var sidebarManager = modelTab.GetSidebarManager();

            Application.Current.Dispatcher.Invoke(() =>
            {
                var pluginWindow = new PluginWindow();
                var viewModel = new PluginViewModel(pluginWindow);
                pluginWindow.DataContext = viewModel;
                var sidebarTab = sidebarManager.AddTab(0, nameof(ModelViewerSample), _icon, pluginWindow);
                viewModel.SidebarTab = sidebarTab;

                if (modelTab.ModelViewer != null)
                {
                    sidebarTab.IsVisible = true;
                    viewModel.OnModelLoaded(modelTab.ModelViewer);
                }
                else
                {
                    sidebarTab.IsVisible = false;
                }

                _tabViewModels.Add(modelTab.Id, viewModel);
            });

            modelTab.Disposed += (_, __) => OnModelTabDisposed(modelTab.Id);
        }

        private void Subscribe()
        {
            _tabsManager.TabOpened += OnTabOpened;
            _modelManager.ModelLoaded += OnModelLoaded;
            _modelManager.ModelClosed += OnModelClosed;
        }

        private void OnTabOpened(object sender, TabEventArgs e)
        {
            if (e.Tab is IModelTab modelTab)
                AddSidebarTab(modelTab);
        }

        private void OnModelLoaded(object sender, ModelEventArgs e)
        {
            _tabViewModels[e.Viewer.TabId].OnModelLoaded(e.Viewer);
        }

        private void OnModelClosed(object sender, ModelEventArgs e)
        {
            _tabViewModels[e.Viewer.TabId].OnModelClosed();
        }

        public void OnModelTabDisposed(Guid id)
        {
            _tabViewModels[id] = null;
            _tabViewModels.Remove(id);
        }
    }
}