﻿using System;
using System.Linq;
using System.Text;
using Prism.Commands;
using System.ComponentModel;
using System.Windows.Input;
using System.Windows.Threading;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Runtime.CompilerServices;
using Ascon.Pilot.Bim.SDK.ModelViewer;
using Ascon.Pilot.Bim.SDK.ModelSamplesHelpers;
using Ascon.Pilot.Bim.SDK.ModelTab.SidebarTab;
using Ascon.Pilot.Bim.SDK.ModelViewerSample.Converters.Parameters;

namespace Ascon.Pilot.Bim.SDK.ModelViewerSample
{
    public class PluginViewModel : INotifyPropertyChanged
    {
        private readonly IUpdateListener _updateListener;
        private IModelViewer _modelViewer;

        private DelegateCommand _getVersionCommand;
        private DelegateCommand<string> _getIsModelPartLoadedCommand;
        private DelegateCommand _getSelectionCommand;
        private DelegateCommand<string> _selectCommand;
        private DelegateCommand _isolateSelectionCommand;
        private DelegateCommand _clearSelectionCommand;
        private DelegateCommand _getVisibleElementsCommand;
        private DelegateCommand _getHiddenElementsCommand;
        private DelegateCommand<string> _getIsVisibleCommand;
        private DelegateCommand<string> _showCommand;
        private DelegateCommand<string> _hideCommand;
        private DelegateCommand<TreeNodeCommandParameters> _expandTreeNodeCommand;
        private DelegateCommand<TreeNodeCommandParameters> _collapseTreeNodeCommand;
        private DelegateCommand<ColorCommandParameters> _setColorCommand;
        private DelegateCommand<string> _clearColorsCommand;
        private DelegateCommand _fitToViewCommand;
        private DelegateCommand<string> _fitToViewForElementsCommand;
        private DelegateCommand _getCameraPositionCommand;
        private DelegateCommand<CameraCommandParameters> _setCameraPositionCommand;
        private DelegateCommand _getClipPlanesCommand;
        private DelegateCommand<ClipPlaneCommandParameters> _addClipPlaneCommand;
        private DelegateCommand<string> _invertClipPlaneCommand;
        private DelegateCommand<string> _removeClipPlaneCommand;
        private DelegateCommand<ScreenshotCommandParameters> _makeScreenshotCommand;
        private DelegateCommand _clearLogCommand;

        public PluginViewModel(IUpdateListener updateListener)
        {
            _updateListener = updateListener;

            Log.CollectionChanged += (sender, args) =>
            {
                if (args.Action != NotifyCollectionChangedAction.Add)
                    return;

                Dispatcher.CurrentDispatcher.BeginInvoke((Action)(() =>
                {
                    _updateListener.OnLogUpdated();
                }));
            };

            InitCommands();
        }

        private void InitCommands()
        {
            _getVersionCommand = new DelegateCommand(GetVersion, () => IsLinked);
            _getIsModelPartLoadedCommand = new DelegateCommand<string>(GetIsModelPartLoaded, x => IsLinked);
            _getSelectionCommand = new DelegateCommand(GetSelection, () => IsLinked);
            _selectCommand = new DelegateCommand<string>(Select, x => IsLinked);
            _isolateSelectionCommand = new DelegateCommand(IsolateSelection, () => IsLinked);
            _clearSelectionCommand = new DelegateCommand(ClearSelection, () => IsLinked);
            _getVisibleElementsCommand = new DelegateCommand(GetVisibleElements, () => IsLinked);
            _getHiddenElementsCommand = new DelegateCommand(GetHiddenElements, () => IsLinked);
            _getIsVisibleCommand = new DelegateCommand<string>(GetIsVisible, x => IsLinked);
            _showCommand = new DelegateCommand<string>(Show, x => IsLinked);
            _hideCommand = new DelegateCommand<string>(Hide, x => IsLinked);
            _expandTreeNodeCommand = new DelegateCommand<TreeNodeCommandParameters>(ExpandTreeNode, x => IsLinked);
            _collapseTreeNodeCommand = new DelegateCommand<TreeNodeCommandParameters>(CollapseTreeNode, x => IsLinked);
            _setColorCommand = new DelegateCommand<ColorCommandParameters>(SetColor, x => IsLinked);
            _clearColorsCommand = new DelegateCommand<string>(ClearColors, x => IsLinked);
            _fitToViewCommand = new DelegateCommand(() => FitToView(), () => IsLinked);
            _fitToViewForElementsCommand = new DelegateCommand<string>(FitToView, x => IsLinked);
            _getCameraPositionCommand = new DelegateCommand(GetCameraPosition, () => IsLinked);
            _setCameraPositionCommand = new DelegateCommand<CameraCommandParameters>(SetCameraPosition, x => IsLinked);
            _getClipPlanesCommand = new DelegateCommand(GetClipPlanes, () => IsLinked);
            _addClipPlaneCommand = new DelegateCommand<ClipPlaneCommandParameters>(AddClipPlane, x => IsLinked);
            _invertClipPlaneCommand = new DelegateCommand<string>(InvertClipPlane, x => IsLinked);
            _removeClipPlaneCommand = new DelegateCommand<string>(RemoveClipPlane, x => IsLinked);
            _makeScreenshotCommand = new DelegateCommand<ScreenshotCommandParameters>(MakeScreenshot, x => IsLinked);
            _clearLogCommand = new DelegateCommand(ClearLog);
        }

        public ObservableCollection<LogEntry> Log { get; } = new ObservableCollection<LogEntry>();

        public ICommand GetVersionCommand => _getVersionCommand;
        public ICommand GetIsModelPartLoadedCommand => _getIsModelPartLoadedCommand;
        public ICommand GetSelectionCommand => _getSelectionCommand;
        public ICommand SelectCommand => _selectCommand;
        public ICommand IsolateSelectionCommand => _isolateSelectionCommand;
        public ICommand ClearSelectionCommand => _clearSelectionCommand;
        public ICommand GetVisibleElementsCommand => _getVisibleElementsCommand;
        public ICommand GetHiddenElementsCommand => _getHiddenElementsCommand;
        public ICommand GetIsVisibleCommand => _getIsVisibleCommand;
        public ICommand ShowCommand => _showCommand;
        public ICommand HideCommand => _hideCommand;
        public ICommand ExpandTreeNodeCommand => _expandTreeNodeCommand;
        public ICommand CollapseTreeNodeCommand => _collapseTreeNodeCommand;
        public ICommand SetColorCommand => _setColorCommand;
        public ICommand ClearColorsCommand => _clearColorsCommand;
        public ICommand FitToViewCommand => _fitToViewCommand;
        public ICommand FitToViewForElementsCommand => _fitToViewForElementsCommand;
        public ICommand GetCameraPositionCommand => _getCameraPositionCommand;
        public ICommand SetCameraPositionCommand => _setCameraPositionCommand;
        public ICommand GetClipPlanesCommand => _getClipPlanesCommand;
        public ICommand AddClipPlaneCommand => _addClipPlaneCommand;
        public ICommand InvertClipPlaneCommand => _invertClipPlaneCommand;
        public ICommand RemoveClipPlaneCommand => _removeClipPlaneCommand;
        public ICommand MakeScreenshotCommand => _makeScreenshotCommand;
        public ICommand ClearLogCommand => _clearLogCommand;

        private bool _isLinked;
        public bool IsLinked
        {
            get => _isLinked;
            set
            {
                _isLinked = value;
                OnPropertyChanged(nameof(IsLinked));
                InvalidateCommands();
            }
        }

        public ISidebarTab SidebarTab { get; set; }

        public bool IsElementsModeEnabled { get; set; } = true;

        private bool _isCameraAnimationChecked;
        public bool IsCameraAnimationChecked
        {
            get => _isCameraAnimationChecked;
            set
            {
                if (_isCameraAnimationChecked == value) 
                    return;

                _isCameraAnimationChecked = value;
                var settingsToChange = _modelViewer.GetDisplaySettings();
                settingsToChange.ModelViewerCommonSettings.UseCameraAnimation = _isCameraAnimationChecked;
                _modelViewer.SetDisplaySettings(settingsToChange);
                OnPropertyChanged(nameof(IsCameraAnimationChecked));
            }
        }

        public void OnModelLoaded(IModelViewer viewer)
        {
            if (ReferenceEquals(_modelViewer, viewer))
                return;

            _modelViewer = viewer;
            SidebarTab.IsVisible = true;
            _modelViewer.ModelVersionAdded += OnModelVersionAdded;
            _modelViewer.ModelVersionChanged += OnModelVersionChanged;
            _modelViewer.SelectionChanged += OnSelectionChanged;
            _modelViewer.TreeNodeExpanded += OnTreeNodeExpanded;
            _modelViewer.TreeNodeCollapsed += OnTreeNodeCollapsed;
            _modelViewer.ModelPartLoadingRequested += OnModelPartLoadingRequested;
            _modelViewer.ModelPartUnloadingRequested += OnModelPartUnloadingRequested;
            IsLinked = true;
            IsCameraAnimationChecked = _modelViewer.GetDisplaySettings().ModelViewerCommonSettings.UseCameraAnimation;
            ClearLog();
            Log.Add(new LogEntry("===> Successfully linked to the model viewer"));
        }

        public void GetVersion()
        {
            var version = _modelViewer.ModelVersion;

            Messages.ShowMessage(version.Ticks.ToString(), "Model version");
        }

        public void GetIsModelPartLoaded(string id)
        {
            var modelPartId = Parsers.GetModelPartId(id);

            if (modelPartId != null)
            {
                var isModelPartLoaded = _modelViewer.IsModelPartLoaded(modelPartId.Value);
                Messages.ShowMessage($"Is model part loaded: {isModelPartLoaded}", "Model part loading state");
            }
        }

        public void GetSelection()
        {
            var selection = _modelViewer.GetSelection().ToList();
            var sb = new StringBuilder(selection.Any() ? $"Selection ({selection.Count} elements):\n" : "Empty selection");

            foreach (var element in selection)
                sb.Append($"Id: {element.ElementId}, model part id: {element.ModelPartId}\n");

            Messages.ShowMessage(sb.ToString(), "Selection");
        }

        public void Select(string ids)
        {
            var elements = Parsers.GetModelElements(ids);

            if (elements.Any())
                _modelViewer.Select(elements);
        }

        public void IsolateSelection()
        {
            _modelViewer.IsolateSelection();
        }

        public void ClearSelection()
        {
            _modelViewer.ClearSelection();
        }

        public void GetVisibleElements()
        {
            var elements = _modelViewer.GetVisibleElements().ToList();
            var sb = new StringBuilder(elements.Any() ? $"Visible elements ({elements.Count} elements):\n" : "No visible elements");

            foreach (var element in elements)
                sb.Append($"Id: {element.ElementId}, model part id: {element.ModelPartId}\n");

            Messages.ShowMessage(sb.ToString(), "Visible elements");
        }

        public void GetHiddenElements()
        {
            var elements = _modelViewer.GetHiddenElements().ToList();
            var sb = new StringBuilder(elements.Any() ? $"Hidden elements ({elements.Count} elements):\n" : "No hidden elements");

            foreach (var element in elements)
                sb.Append($"Id: {element.ElementId}, model part id: {element.ModelPartId}\n");

            Messages.ShowMessage(sb.ToString(), "Hidden elements");
        }

        public void GetIsVisible(string ids)
        {
            var elements = Parsers.GetModelElements(ids).ToList();

            foreach (var element in elements)
            {
                var isVisible = _modelViewer.IsVisible(element);

                var visibility = isVisible ? "visible" : "hidden";
                Messages.ShowMessage($"Id: {element.ElementId}, model part id: {element.ModelPartId}, visibility: {visibility}", "Element visibility");
            }
        }

        public void Show(string ids)
        {
            if (IsElementsModeEnabled)
            {
                var elements = Parsers.GetModelElements(ids);

                if (elements.Any())
                    _modelViewer.Show(elements);
            }
            else
            {
                var modelPartId = Parsers.GetModelPartId(ids);

                if (modelPartId != null)
                    _modelViewer.Show(modelPartId.Value);
            }
        }

        public void Hide(string ids)
        {
            if (IsElementsModeEnabled)
            {
                var elements = Parsers.GetModelElements(ids);

                if (elements.Any())
                    _modelViewer.Hide(elements);
            }
            else
            {
                var modelPartId = Parsers.GetModelPartId(ids);

                if (modelPartId != null)
                    _modelViewer.Hide(modelPartId.Value);
            }
        }

        public void ExpandTreeNode(TreeNodeCommandParameters parameters)
        {
            var elements = Parsers.GetModelElements(parameters.Ids);
            var levelsValue = GetLevels(parameters.Levels);

            if (levelsValue == null)
                return;

            foreach (var element in elements)
            {
                _modelViewer.ExpandTreeNode(element, levelsValue.Value);
            }
        }

        public void CollapseTreeNode(TreeNodeCommandParameters parameters)
        {
            var elements = Parsers.GetModelElements(parameters.Ids);
            var levelsValue = GetLevels(parameters.Levels);

            if (levelsValue == null)
                return;

            foreach (var element in elements)
            {
                _modelViewer.CollapseTreeNode(element, levelsValue.Value);
            }
        }

        public void SetColor(ColorCommandParameters parameters)
        {
            var color = GetColor(parameters.R, parameters.G, parameters.B, parameters.A);

            if (color == null)
                return;

            if (IsElementsModeEnabled)
            {
                var elements = Parsers.GetModelElements(parameters.Ids);

                if (elements.Any())
                    _modelViewer.SetColor(elements, color.Value);
            }
            else
            {
                var modelPartId = Parsers.GetModelPartId(parameters.Ids);

                if (modelPartId != null)
                    _modelViewer.SetColor(modelPartId.Value, color.Value);
            }
        }

        public void ClearColors(string ids)
        {
            if (IsElementsModeEnabled)
            {
                var elements = Parsers.GetModelElements(ids);

                if (elements.Any())
                    _modelViewer.ClearColors(elements);
            }
            else
            {
                var modelPartId = Parsers.GetModelPartId(ids);

                if (modelPartId != null)
                    _modelViewer.ClearColors(modelPartId.Value);
            }
        }

        public void FitToView(string ids = null)
        {
            if (ids == null)
            {
                _modelViewer.FitToView();
            }
            else
            {
                if (IsElementsModeEnabled)
                {
                    var elements = Parsers.GetModelElements(ids);

                    if (elements.Any())
                        _modelViewer.FitToView(elements);
                }
                else
                {
                    var modelPartId = Parsers.GetModelPartId(ids);

                    if (modelPartId != null)
                        _modelViewer.FitToView(modelPartId.Value);
                }
            }
        }

        public void GetCameraPosition()
        {
            var cameraPosition = _modelViewer.GetCameraPosition2();
            var position = $"Camera position:\n" +
                           $"X: {cameraPosition.X}\n" +
                           $"Y: {cameraPosition.Y}\n" +
                           $"Z: {cameraPosition.Z}\n" +
                           $"EyeX: {cameraPosition.EyeX}\n" +
                           $"EyeY: {cameraPosition.EyeY}\n" +
                           $"EyeZ: {cameraPosition.EyeZ}\n" +
                           $"PivotX: {cameraPosition.PivotX}\n" +
                           $"PivotY: {cameraPosition.PivotY}\n" +
                           $"PivotZ: {cameraPosition.PivotZ}\n" +
                           $"Angle: {cameraPosition.Angle}\n";

            Messages.ShowMessage(position, "Camera position");
        }

        public void SetCameraPosition(CameraCommandParameters parameters)
        {
            double cameraX = 0,
                cameraY = 0,
                cameraZ = 0;
            float cameraEyeX = 0,
                cameraEyeY = 0,
                cameraEyeZ = 0;
            double cameraPivotX = 0,
                cameraPivotY = 0,
                cameraPivotZ = 0;
            float cameraAngle = 0;

            var result = double.TryParse(parameters.X, out cameraX) &&
                         double.TryParse(parameters.Y, out cameraY) &&
                         double.TryParse(parameters.Z, out cameraZ) &&
                         float.TryParse(parameters.EyeX, out cameraEyeX) &&
                         float.TryParse(parameters.EyeY, out cameraEyeY) &&
                         float.TryParse(parameters.EyeZ, out cameraEyeZ) &&
                         double.TryParse(parameters.PivotX, out cameraPivotX) &&
                         double.TryParse(parameters.PivotY, out cameraPivotY) &&
                         double.TryParse(parameters.PivotZ, out cameraPivotZ) &&
                         float.TryParse(parameters.Angle, out cameraAngle);

            if (!result)
                Messages.ShowErrorMessage("Invalid camera position");
            else
            {
                var cameraPosition = new CameraPosition(cameraX, cameraY, cameraZ, cameraEyeX, cameraEyeY, cameraEyeZ,
                    cameraPivotX, cameraPivotY, cameraPivotZ, cameraAngle);
                _modelViewer.SetCameraPosition2(cameraPosition);
            }
        }

        public void GetClipPlanes()
        {
            var clipPlanes = _modelViewer.GetClipPlanes().ToList();
            var sb = new StringBuilder(clipPlanes.Any() ? $"Clip planes ({clipPlanes.Count}):\n" : "No clip planes");

            foreach (var clipPlane in clipPlanes)
                sb.Append($"Id: {clipPlane.Id}, X: {clipPlane.Properties.X}, Y: {clipPlane.Properties.Y}, Z: {clipPlane.Properties.Z}, " +
                          $"NormalX: {clipPlane.Properties.NormalX}, NormalY: {clipPlane.Properties.NormalY}, NormalZ: {clipPlane.Properties.NormalZ}\n");

            Messages.ShowMessage(sb.ToString(), "Clip planes");
        }

        public void AddClipPlane(ClipPlaneCommandParameters parameters)
        {
            float clipPlaneX = 0,
                clipPlaneY = 0,
                clipPlaneZ = 0,
                clipPlaneNormalX = 0,
                clipPlaneNormalY = 0,
                clipPlaneNormalZ = 0;

            var result = float.TryParse(parameters.X, out clipPlaneX) &&
                         float.TryParse(parameters.Y, out clipPlaneY) &&
                         float.TryParse(parameters.Z, out clipPlaneZ) &&
                         float.TryParse(parameters.NormalX, out clipPlaneNormalX) &&
                         float.TryParse(parameters.NormalY, out clipPlaneNormalY) &&
                         float.TryParse(parameters.NormalZ, out clipPlaneNormalZ);

            if (!result)
                Messages.ShowErrorMessage("Invalid clip plane properties");
            else
            {
                var clipPlaneProperties = new ClipPlaneProperties(clipPlaneX, clipPlaneY, clipPlaneZ, clipPlaneNormalX, clipPlaneNormalY, clipPlaneNormalZ);
                var clipPlaneId = _modelViewer.AddClipPlane(clipPlaneProperties);
                Messages.ShowMessage($"Clip plane id: {clipPlaneId}", "Clip plane added");
            }
        }

        public void InvertClipPlane(string id)
        {
            var clipPlaneId = GetClipPlaneId(id);

            if (clipPlaneId != null)
                _modelViewer.InvertClipPlane(clipPlaneId.Value);
        }

        public void RemoveClipPlane(string id)
        {
            var clipPlaneId = GetClipPlaneId(id);

            if (clipPlaneId != null)
                _modelViewer.RemoveClipPlane(clipPlaneId.Value);
        }

        public void MakeScreenshot(ScreenshotCommandParameters parameters)
        {
            if (int.TryParse(parameters.Width, out var widthValue))
            {
                if (int.TryParse(parameters.Height, out var heightValue))
                {
                    var screenshot = _modelViewer.MakeScreenshot(widthValue, heightValue);
                    _updateListener.OnScreenshotUpdated(screenshot);
                }
            }
            else
            {
                Messages.ShowErrorMessage("Invalid screenshot size");
            }
        }

        private void OnModelVersionAdded(object sender, ModelVersionEventArgs e)
        {
            Log.Add(new LogEntry($"===> Version added: {e.Version.Ticks}"));
        }

        private void OnModelVersionChanged(object sender, ModelVersionEventArgs e)
        {
            Log.Add(new LogEntry($"===> Version changed: {e.Version.Ticks}"));
        }

        public void OnSelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            var elements = e.SelectedIds.ToList();

            if (!elements.Any())
                Log.Add(new LogEntry("===> Selection cleared"));
            else
                Log.Add(new LogEntry($"===> Selection changed ({elements.Count} elements)"));
        }

        public void OnTreeNodeExpanded(object sender, TreeNodeEventArgs e)
        {
            Log.Add(new LogEntry($"===> Element expanded (id: {e.Id.ElementId}, model part id: {e.Id.ModelPartId})"));
        }

        public void OnTreeNodeCollapsed(object sender, TreeNodeEventArgs e)
        {
            Log.Add(new LogEntry($"===> Element collapsed (id: {e.Id.ElementId}, model part id: {e.Id.ModelPartId})"));
        }

        public void OnModelPartLoadingRequested(object sender, ModelPartEventArgs e)
        {
            Log.Add(new LogEntry($"===> Model part loading requested, model part id: {e.Id})"));
        }

        public void OnModelPartUnloadingRequested(object sender, ModelPartEventArgs e)
        {
            Log.Add(new LogEntry($"===> Model part unloading requested, model part id: {e.Id})"));
        }

        public void ClearLog()
        {
            Log.Clear();
        }

        public void OnModelClosed()
        {
            if (_modelViewer == null)
                return;

            SidebarTab.IsVisible = false;
            _modelViewer.ModelVersionAdded -= OnModelVersionAdded;
            _modelViewer.ModelVersionChanged -= OnModelVersionChanged;
            _modelViewer.SelectionChanged -= OnSelectionChanged;
            _modelViewer.TreeNodeExpanded -= OnTreeNodeExpanded;
            _modelViewer.TreeNodeCollapsed -= OnTreeNodeCollapsed;
            _modelViewer.ModelPartLoadingRequested -= OnModelPartLoadingRequested;
            _modelViewer.ModelPartUnloadingRequested -= OnModelPartUnloadingRequested;
            _modelViewer = null;
        }

        private void InvalidateCommands()
        {
            _getVersionCommand.RaiseCanExecuteChanged();
            _getIsModelPartLoadedCommand.RaiseCanExecuteChanged();
            _getSelectionCommand.RaiseCanExecuteChanged();
            _selectCommand.RaiseCanExecuteChanged();
            _isolateSelectionCommand.RaiseCanExecuteChanged();
            _clearSelectionCommand.RaiseCanExecuteChanged();
            _getVisibleElementsCommand.RaiseCanExecuteChanged();
            _getHiddenElementsCommand.RaiseCanExecuteChanged();
            _getIsVisibleCommand.RaiseCanExecuteChanged();
            _showCommand.RaiseCanExecuteChanged();
            _hideCommand.RaiseCanExecuteChanged();
            _expandTreeNodeCommand.RaiseCanExecuteChanged();
            _collapseTreeNodeCommand.RaiseCanExecuteChanged();
            _setColorCommand.RaiseCanExecuteChanged();
            _clearColorsCommand.RaiseCanExecuteChanged();
            _fitToViewCommand.RaiseCanExecuteChanged();
            _fitToViewForElementsCommand.RaiseCanExecuteChanged();
            _getCameraPositionCommand.RaiseCanExecuteChanged();
            _setCameraPositionCommand.RaiseCanExecuteChanged();
            _getClipPlanesCommand.RaiseCanExecuteChanged();
            _addClipPlaneCommand.RaiseCanExecuteChanged();
            _invertClipPlaneCommand.RaiseCanExecuteChanged();
            _removeClipPlaneCommand.RaiseCanExecuteChanged();
            _makeScreenshotCommand.RaiseCanExecuteChanged();
        }

        private static int? GetLevels(string levels)
        {
            if (int.TryParse(levels, out var levelsValue) && levelsValue >= 0)
                return levelsValue;

            Messages.ShowErrorMessage("Levels value is invalid (should be >= 0)");
            return null;
        }

        private static Color? GetColor(string r, string g, string b, string a)
        {
            if (byte.TryParse(r, out var rComponent))
            {
                if (byte.TryParse(g, out var gComponent))
                {
                    if (byte.TryParse(b, out var bComponent))
                    {
                        if (byte.TryParse(a, out var aComponent))
                        {
                            return new Color(rComponent, gComponent, bComponent, aComponent);
                        }
                    }
                }
            }

            Messages.ShowErrorMessage("Invalid color components (should be 0 - 255)");
            return null;
        }

        private static uint? GetClipPlaneId(string id)
        {
            if (uint.TryParse(id, out var clipPlaneId))
            {
                return clipPlaneId;
            }

            Messages.ShowErrorMessage("Clip plane id value is invalid (should be >= 0)");
            return null;
        }

        public event PropertyChangedEventHandler PropertyChanged;

        protected virtual void OnPropertyChanged([CallerMemberName] string propertyName = "")
        {
            PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(propertyName));
        }
    }
}