/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Menu;
using Ascon.Pilot.SDK.Toolbar;
using ChangeChildrenStateActivity;
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Windows.Media;

namespace AnnotationPageChanger
{
    [Export(typeof(IMenu<DocumentAnnotationsListContext>))]
    [Export(typeof(IToolbar<XpsRenderContext>))]
    public class AnnotationPageChanger : IMenu<DocumentAnnotationsListContext>, IToolbar<XpsRenderContext>, IMouseLeftClickListener
    {
        private const string TEXT_NOTE_NAME = "TextNote";
        private const string TEXT_STICKY_NOTE_NAME = "TextStickyNote";
        private const string RED_PENCIL_NAME = "RedPencil";
        private const string CHANGE_ANNOTATION_PAGE_NAME = "changeAnnotationPage";
        private const string ANNOTATION_ATTRIBUTE_NAME = "annotation";

        private IObjectModifier _modifier;
        private readonly IXpsViewer _viewer;
        private readonly IObjectsRepository _repository;
        private readonly IPerson _person;
        private DateTime _curentVersion;
        private IDataObject _document;
        private IDataObject _annotationToCut;

        [ImportingConstructor]
        public AnnotationPageChanger(IObjectModifier modifier, IXpsViewer viewer, IObjectsRepository repository)
        {
            _modifier = modifier;
            _viewer = viewer;
            _repository = repository;
            _person = _repository.GetCurrentPerson();
        }

        public void Build(IMenuBuilder builder, DocumentAnnotationsListContext context)
        {
            if (CanCutAnnotation(context))
                builder.AddItem(CHANGE_ANNOTATION_PAGE_NAME, builder.Count).WithHeader("Перенести замечание на другую страницу");
        }

        public void OnMenuItemClick(string name, DocumentAnnotationsListContext context)
        {
            if (name == CHANGE_ANNOTATION_PAGE_NAME)
            {
                _annotationToCut = context.SelectedAnnotations.First();
                _viewer.SubscribeLeftMouseClick(this);
            }
        }

        public void OnLeftMouseButtonClick(XpsRenderClickPointContext pointContext)
        {
            if (_annotationToCut == null)
                return;

            var container = GetActualAnnotationForCurentVersion(_annotationToCut);

            if (container.Kind == RED_PENCIL_NAME)
                TransformGeometry(container, pointContext.ClickPoint);

            container.PageNumber = pointContext.PageNumber;
            container.PositionX = pointContext.ClickPoint.X;
            container.PositionY = pointContext.ClickPoint.Y;
            container.Version = _curentVersion;
            SaveAnnotation(_annotationToCut, container);

            _annotationToCut = null;
            _viewer.UnsubscribeLeftMouseClick(this);
        }

        public void Build(IToolbarBuilder builder, XpsRenderContext context)
        {
            _curentVersion = context.SelectedVersion;
            _document = context.DataObject;
        }

        private bool CanCutAnnotation(DocumentAnnotationsListContext context)
        {
            if (context.SelectedAnnotations.Count() != 1) //can use only on one annotation
                return false;

            var obj = context.SelectedAnnotations.First();
            var access = _repository.GetCurrentAccess(obj.Id, _person.Id);
            if (!access.HasFlag(AccessLevel.Edit)) //should have edit access for object
                return false;

            var attr = obj.Type.Attributes.First(x => x.Name == ANNOTATION_ATTRIBUTE_NAME);
            var pi = new AttributePermissionInfo(attr);
            if (pi.ExtractAllOrgUnits(obj).Intersect(_person.AllOrgUnits()).Any() == false)
                return false; //should have edit access for attribute

            if (_curentVersion != _document.ActualFileSnapshot.Created) //document should be open in actual version
                return false;

            var annotaton = GetActualAnnotationForCurentVersion(obj);

            if (!IsAnnotationActualForCurrentVersion(obj)) //annotation should be in actual version
                return false;

            if (annotaton.Kind == TEXT_STICKY_NOTE_NAME) //can't move text sticky annotation
                return false;

            return true;
        }
        private XpsAnnotation GetActualAnnotationForCurentVersion(IDataObject obj)
        {
            using (var reader = new StringReader(obj.Attributes[ANNOTATION_ATTRIBUTE_NAME] as string))
            {
                var containers = AnnotationSerializers.AnnotationSerializer.Deserialize(reader) as List<XpsAnnotation>;
                return containers?.OrderByDescending(c => c.Version).FirstOrDefault(c => c.Version <= _curentVersion);
            }
        }

        private bool IsAnnotationActualForCurrentVersion(IDataObject obj)
        {
            return obj.Attributes.Any(x => x.Key == ANNOTATION_ATTRIBUTE_NAME + _curentVersion.Ticks);
        }

        private void SaveAnnotation(IDataObject obj, XpsAnnotation annotation)
        {
            using (var reader = new StringReader(obj.Attributes[ANNOTATION_ATTRIBUTE_NAME] as string))
            {
                var containers = AnnotationSerializers.AnnotationSerializer.Deserialize(reader) as List<XpsAnnotation>;

                var existingIndex = containers.FindIndex(c => c.Version == _curentVersion);
                if (existingIndex != -1)
                    containers[existingIndex] = annotation;
                else
                    containers.Add(annotation);

                var builder = _modifier.Edit(obj);
                builder.SetAttribute(ANNOTATION_ATTRIBUTE_NAME, AnnotationSerializers.SerializeToString(containers));
                _modifier.Apply();
            }
        }

        private void TransformGeometry(XpsAnnotation container, System.Windows.Point point)
        {
            using (var strReader = new StringReader(container.Data))
            {
                var pencilData = PencilDataSerializer.Serializer.Deserialize(strReader) as PencilData;
                var geometry = Geometry.Parse(pencilData.Geometry).CloneCurrentValue();
                geometry.Transform = new TranslateTransform(point.X - geometry.Bounds.X, point.Y - geometry.Bounds.Y);
                pencilData.Geometry = geometry.GetFlattenedPathGeometry().ToString(CultureInfo.InvariantCulture);
                container.Data = PencilDataSerializer.SerializeToString(pencilData);
            }
        }

        public void OnToolbarItemClick(string name, XpsRenderContext context)
        {
        }
    }

    [Serializable]
    public class PencilData
    {
        public string Geometry { get; set; }
        public string Color { get; set; }
        public bool IsStraightLine { get; set; }
        public bool IsTextNoteVisible { get; set; }

    }
}
