/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Security.Cryptography.Pkcs;
using System.Security.Cryptography.X509Certificates;
using Ascon.Pilot.Theme.Controls.Windows.Tools;
using Ascon.Pilot.Theme.Tools;

namespace Ascon.Pilot.SDK.DocumentCryptoProviderSample
{
#pragma warning disable CS0618 // Type or member is obsolete
    [Export(typeof(IDocumentCryptoProvider))]
    public class DocumentCryptoProvider : IDocumentCryptoProvider
#pragma warning restore CS0618 // Type or member is obsolete
    {
        private const string SIG_EXTENSION = ".sig";
        private const string XPS_EXTENSION = ".xps";

        private readonly IObjectModifier _modifier;
        private readonly IFileProvider _fileProvider;

        [ImportingConstructor]
        public DocumentCryptoProvider(IObjectModifier modifier, IFileProvider fileProvider)
        {
            _modifier = modifier;
            _fileProvider = fileProvider;
        }

        public bool IsKnownSignature(IDataObject document, IFilesSnapshot snapshot, ISignatureRequest signatureRequest)
        {
            if(!signatureRequest.Signs.Any() || !Guid.TryParse(signatureRequest.Signs.First(), out var signId))
                return false;

            var signFile = snapshot.Files.FirstOrDefault(x => x.Id == signId);
            if (signFile == null)
                return false;

            return signFile.Name.EndsWith(SIG_EXTENSION);
        }

        public void Sign(IDataObject document, Stream xpsStream, IEnumerable<ISignatureRequest> signatureRequests)
        {
            var certificate = SelectCertificate();
            _modifier.Clear();
            var builder = _modifier.Edit(document);
            
            foreach (var signatureRequest in signatureRequests)
            {
                foreach (var file in document.Files.Where(ShouldBeSigned))
                {
                    bool isXps = file.Name.EndsWith(XPS_EXTENSION, StringComparison.InvariantCultureIgnoreCase);
                    using (var documentStream = _fileProvider.OpenRead(file))
                    {
                        var documentData = StreamToArray(isXps ? xpsStream : documentStream);
                        ContentInfo contentInfo = new ContentInfo(documentData);
                        SignedCms cms = new SignedCms(contentInfo, true);
                        CmsSigner signer = new CmsSigner(certificate);
                        cms.ComputeSignature(signer, false);
                        var signatureData = cms.Encode();
                        using (var ms = new MemoryStream(signatureData))
                        {
                            builder.AddFile($"{file.Name}.{signatureRequest.Id}{SIG_EXTENSION}", ms, DateTime.UtcNow,
                                DateTime.UtcNow, DateTime.UtcNow, out var fileId);
                            if (isXps)
                                builder.SetSignatures(f => f.Id == file.Id).SetSign(signatureRequest.Id, fileId.ToString());
                        }
                    }
                }
            }
            _modifier.Apply();
        }

        public void ReadSignature(IDataObject document, Stream xpsStream, IFilesSnapshot snapshot, ISignatureRequest signatureRequest, IReadSignatureListener listener)
        {
            try
            {
                if (!signatureRequest.Signs.Any() || !Guid.TryParse(signatureRequest.Signs.First(), out var signId))
                    return;

                var signFile = snapshot.Files.FirstOrDefault(x => x.Id == signId);
                if (signFile == null)
                    return;

                var documentFile = snapshot.Files.FirstOrDefault(f => f.SignatureRequests.Any(s => s.Id == signatureRequest.Id));
                if (documentFile == null)
                    return;
                
                using (var signatureStream = _fileProvider.OpenRead(signFile))
                {
                    var documentData = StreamToArray(xpsStream);
                    var signatureData = StreamToArray(signatureStream);

                    ContentInfo contentInfo = new ContentInfo(documentData);
                    SignedCms cms = new SignedCms(contentInfo, true);

                    try
                    {
                        cms.Decode(signatureData);
                        cms.CheckSignature(true);

                        listener.SignatureIsValid();
                        var signer = cms.SignerInfos[0];
                        
                        if(CheckCertificate(signer.Certificate))
                            listener.CertificateIsValid();

                        var signDate = GetSignDate(signer);
                        if (signDate != null)
                            listener.SetSignDate(signDate.Value.ToLocalTime().ToString("g"));

                        var subject = ParseCertificateSubject(signer.Certificate.Subject);
                        if(subject.TryGetValue("CN", out var subjectCn))
                            listener.SetSignerName("Подписан: " + subjectCn);
                    }
                    catch (Exception e)
                    {
                        listener.SignatureIsInvalid(e.Message);
                    }

                }
            }
            finally
            {
                listener.ReadCompleted();
            }
        }

        public void ShowCertificate(IDataObject document, IFilesSnapshot snapshot, ISignatureRequest signatureRequest)
        {
            if (!signatureRequest.Signs.Any() || !Guid.TryParse(signatureRequest.Signs.First(), out var signId))
                return;

            var signFile = snapshot.Files.FirstOrDefault(x => x.Id == signId);
            if (signFile == null)
                return;

            var documentFile =
                snapshot.Files.FirstOrDefault(f => f.SignatureRequests.Any(s => s.Id == signatureRequest.Id));
            if (documentFile == null)
                return;

            using (var documentStream = _fileProvider.OpenRead(documentFile))
            using (var signatureStream = _fileProvider.OpenRead(signFile))
            {
                var documentData = StreamToArray(documentStream);
                var signatureData = StreamToArray(signatureStream);
                ContentInfo contentInfo = new ContentInfo(documentData);
                SignedCms cms = new SignedCms(contentInfo, true);

                try
                {
                    cms.Decode(signatureData);
                    cms.CheckSignature(true);

                    var signer = cms.SignerInfos[0];
                    var certificateRawData = signer.Certificate.GetRawCertData();

                    var ownerWindow = LayoutHelper.GetOwnerWindow();
                    X509Certificate2UI.DisplayCertificate(new X509Certificate2(certificateRawData), ownerWindow.GetWindowHandle());
                }
                catch
                {
                    // do nothing
                }
            }
        }

        private bool ShouldBeSigned(IFile file)
        {
            return file.Name.EndsWith(".xps") || file.Name.EndsWith(".pdf");
        }

        private byte[] StreamToArray(Stream stream)
        {
            using (var ms = new MemoryStream())
            {
                stream.CopyTo(ms);
                return ms.ToArray();
            }
        }

        public X509Certificate2 SelectCertificate()
        {
            var certificateCollection = GetCertificatesCollection();

            if (certificateCollection.Count == 0)
                throw new InvalidOperationException("not certificate");

            if (certificateCollection.Count > 1)
                throw new InvalidOperationException("more than one certificate available");

            return certificateCollection[0];
        }

        private static X509Certificate2Collection GetCertificatesCollection()
        {
            var store = new X509Store(StoreLocation.CurrentUser);
            store.Open(OpenFlags.OpenExistingOnly | OpenFlags.ReadOnly);
            var certificateCollection = store.Certificates;
            store.Close();

            certificateCollection = certificateCollection.Find(X509FindType.FindByTimeValid, DateTime.Now, true);
            certificateCollection = certificateCollection.Find(X509FindType.FindByKeyUsage, X509KeyUsageFlags.DigitalSignature, false);
            var withPrivate = new X509Certificate2Collection();
            foreach (X509Certificate2 cer in certificateCollection.Cast<X509Certificate2>().Where(cer => cer.HasPrivateKey))
                withPrivate.Add(cer);
            certificateCollection = withPrivate;

            return certificateCollection;
        }


        private static IReadOnlyDictionary<string, string> ParseCertificateSubject(string subject)
        {
            var result = subject.Split(new[] { ',' }, StringSplitOptions.RemoveEmptyEntries)
                .Select(x => x.Split(new[] { '=' }, StringSplitOptions.RemoveEmptyEntries))
                .ToList();

            var dict = new Dictionary<string, string>();
            foreach (var t in result)
            {
                if (t.Length != 2)
                    continue;

                var key = t[0].Trim();
                var value = t[1].Trim();

                if (!dict.ContainsKey(key))
                    dict.Add(key, value);
            }
            return dict;
        }

        private DateTime? GetSignDate(SignerInfo si)
        {
            try
            {
                foreach (var attribute in si.SignedAttributes)
                {
                    if (attribute.Oid.Value == "1.2.840.113549.1.9.5") // Oid время подписания
                    {
                        Pkcs9SigningTime pkcs9_time = new Pkcs9SigningTime(attribute.Values[0].RawData);
                        return pkcs9_time.SigningTime;
                    }
                }
            }
            catch
            {
                return null;
            }
            return null;
        }

        private bool CheckCertificate(X509Certificate certificate)
        {
            var x5092Cert = new X509Certificate2(certificate);
            var chain = new X509Chain
            {
                ChainPolicy =
                {
                    RevocationMode = X509RevocationMode.NoCheck,
                    VerificationFlags = X509VerificationFlags.AllowUnknownCertificateAuthority | X509VerificationFlags.IgnoreNotTimeValid
                }
            };
            var isSoftCertificateValidation = chain.Build(x5092Cert);
            var isCertificateValid = false;
            if (isSoftCertificateValidation)
            {
                chain.Reset();
                chain.ChainPolicy.RevocationMode = X509RevocationMode.Online;
                chain.ChainPolicy.VerificationFlags = X509VerificationFlags.NoFlag;
                isCertificateValid = chain.Build(x5092Cert);
            }
            return isSoftCertificateValidation && isCertificateValid;
        }
    }
}
