/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Drawing.Imaging;
using System.Globalization;
using System.IO;
using System.Windows;
using System.Xml.Linq;
using Ascon.Pilot.SDK.GraphicLayerSample;
using ZXing;
using ZXing.QrCode;

namespace Ascon.Pilot.SDK.QRCodeSample
{
    public class QrCode
    {
        public static string PilotUri = "piloturi://";
        public static string WebFormat = "{0}/Files/Index/{1}?version={2}";
        public static string DesktopFormat = "{0}{1}/{2}";

        private readonly HorizontalAlignment _horizontalAlignment;
        private readonly VerticalAlignment _verticalAlignment;

        private readonly string _url;
        private readonly double _offsetX;
        private readonly double _offsetY;
        private readonly double _scale;
        private readonly double _angle;

        private readonly int _height;
        private readonly int _width;
        private readonly int _borderThickness;
        private readonly int _pageNumber;

        public QrCode(XElement qrCode)
        {
            _url = ParseAttribute(qrCode, Constants.URL, PilotUri);
            _offsetX = Convert.ToDouble(ParseAttribute(qrCode, Constants.OFFSET_X, "0"), CultureInfo.InvariantCulture);
            _offsetY = Convert.ToDouble(ParseAttribute(qrCode, Constants.OFFSET_Y, "0"), CultureInfo.InvariantCulture);
            _width = Convert.ToInt32(ParseAttribute(qrCode, Constants.WIDTH, "0"), CultureInfo.InvariantCulture);
            _height = Convert.ToInt32(ParseAttribute(qrCode, Constants.HEIGHT, "0"), CultureInfo.InvariantCulture);
            _angle = Convert.ToDouble(ParseAttribute(qrCode, Constants.ANGLE, "0"), CultureInfo.InvariantCulture);
            _scale = Convert.ToDouble(ParseAttribute(qrCode, Constants.SCALE, "1"), CultureInfo.InvariantCulture);
            _borderThickness = Convert.ToInt32(ParseAttribute(qrCode, Constants.BORDER_THICKNESS, "1"), CultureInfo.InvariantCulture);
            _pageNumber = Convert.ToInt32(ParseAttribute(qrCode, Constants.PAGE_NUMBER, "0"), CultureInfo.InvariantCulture);

            Enum.TryParse(ParseAttribute(qrCode, Constants.VERTICAL_ALIGNMENT, "Top"), out _verticalAlignment);
            Enum.TryParse(ParseAttribute(qrCode, Constants.HORIZONTAL_ALIGNMENT, "Left"), out _horizontalAlignment);
        }

        public GraphicLayerElement GetGraphicLayerElement(Guid elementId, Guid contentId)
        {
            var dpi = 96;
            var xOffset = _offsetX / 25.4 * dpi;
            var yOffset = _offsetY / 25.4 * dpi;

            return new GraphicLayerElement(elementId,
                                           contentId,
                                           yOffset,
                                           xOffset,
                                           0,//if positionId == 0, all people can change(drag and resize) this GraphicLayerElement
                                           new Point(_scale, _scale),
                                           _angle,
                                           _verticalAlignment,
                                           _horizontalAlignment,
                                           GraphicLayerElementConstants.BITMAP,
                                           _pageNumber,
                                           true);
        }

        public Stream GetBitmapStream(Guid objId, long versionDate)
        {
            var options = new QrCodeEncodingOptions
            {
                Width = _width,
                Height = _height,
                PureBarcode = false
            };
            options.Hints[EncodeHintType.MARGIN] = _borderThickness;
            IBarcodeWriter writer = new BarcodeWriter
            {
                Format = BarcodeFormat.QR_CODE,
                Options = options
            };
            var content = string.Format(_url.Equals(PilotUri) ? DesktopFormat : WebFormat, _url, objId, versionDate);
            var bitmap = writer.Write(content);

            var imageStream = new MemoryStream();
            bitmap.Save(imageStream, ImageFormat.Png);
            return imageStream;
        }

        private string ParseAttribute(XElement configuration, string attributeName, string defaultValue)
        {
            var offsetXAttr = configuration.Attribute(attributeName);
            return offsetXAttr?.Value ?? defaultValue;
        }

    }
}