/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Diagnostics;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Ascon.Pilot.SDK.Controls;
using Ascon.Pilot.SDK.Menu;

namespace Ascon.Pilot.SDK.StorageSample
{
    [Export(typeof(IMenu<SystemTrayContext>))]
    public class TrayMenu : IMenu<SystemTrayContext>
    {
        private const string MI_MOUNT_PROJECT = "miMountProjectRoot";
        private const string MI_MOUNT_PROJECT_BY_ID = "miMountProject_";

        private readonly ProjectExplorer _projectExplorer;
        private readonly IObjectsRepository _repository;

        [ImportingConstructor]
        public TrayMenu(IObjectsRepository repository, ISearchService searchService)
        {
            _repository = repository;
            _projectExplorer = new ProjectExplorer(repository, searchService);
        }

        public void Build(IMenuBuilder builder, SystemTrayContext context)
        {
            var item = builder.AddItem(MI_MOUNT_PROJECT, builder.Count - 2);
            var submenu = item.WithHeader("Смонтировать проект")
                              .WithSubmenu();

            int index = 0;
            var projects = _projectExplorer.GetProjects();
            foreach (var project in projects)
            {
                submenu.AddItem(MI_MOUNT_PROJECT_BY_ID + FromGuid(project.Id), index++)
                    .WithHeader(project.Title);
            }
        }

        public async void OnMenuItemClick(string name, SystemTrayContext context)
        {
            if(!name.StartsWith(MI_MOUNT_PROJECT_BY_ID))
                return;

            var id = ToGuid(name.Substring(MI_MOUNT_PROJECT_BY_ID.Length));
            _repository.Mount(id);
            var path = await WaitForStoragePathAsync(id);
            Process.Start("Explorer", @"/e,""" + path + @"""");
        }

        private Task<string> WaitForStoragePathAsync(Guid id, int timeout = 5000)
        {
            return Task<string>.Factory.StartNew(() =>
            {
                Stopwatch sw = Stopwatch.StartNew();
                while (true)
                {
                    if(sw.ElapsedMilliseconds > timeout)
                        throw new TimeoutException();
                    var path = _repository.GetStoragePath(id);
                    if (!string.IsNullOrEmpty(path))
                        return path;
                    Thread.Sleep(10);
                }
            });
        }

        private static string FromGuid(Guid id)
        {
            return id.ToString().Replace("-", "_");
        }

        private static Guid ToGuid(string id)
        {
            return new Guid(id.Replace("_", "-"));
        }

        class Project
        {
            public Guid Id { get; }
            public string Title { get; }

            public Project(Guid id, string title)
            {
                Id = id;
                Title = title;
            }
        }

        class ProjectExplorer : IObserver<ISearchResult>
        {
            private readonly IObjectsRepository _repository;
            private readonly ConcurrentDictionary<Guid, string> _projects = new ConcurrentDictionary<Guid, string>();
            private readonly IDisposable _subscription;

            public ProjectExplorer(IObjectsRepository repository, ISearchService searchService)
            {
                _repository = repository;

                var projectType = repository.GetType("project");
                var builder = searchService.GetObjectQueryBuilder();
                builder.Must(ObjectFields.TypeId.Be(projectType.Id));
                _subscription = searchService.Search(builder).Subscribe(this);
            }

            public IEnumerable<Project> GetProjects() => _projects.Select(x => new Project(x.Key, x.Value)).OrderBy(x => x.Title).ToList();

            async void IObserver<ISearchResult>.OnNext(ISearchResult value)
            {
                if(value.Kind == SearchResultKind.Remote)
                    _subscription.Dispose();

                foreach (var id in value.Result)
                {
                    if(_projects.ContainsKey(id))
                        continue;

                    var loader = new ObjectLoader(_repository);
                    var obj = await loader.Load(id);
                    _projects.TryAdd(id, obj.DisplayName);
                }
            }

            void IObserver<ISearchResult>.OnCompleted()
            {
            }

            void IObserver<ISearchResult>.OnError(Exception error)
            {
            }
        }
    }
}
