/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Windows.Media;
using Ascon.Pilot.SDK.Menu;
using Ascon.Pilot.Theme.ColorScheme;

namespace Ascon.Pilot.SDK.XpsToBitmapRendererSample
{
    [Export(typeof(IMenu<ObjectsViewContext>))]
    public class XpsRendererPlugin : IMenu<ObjectsViewContext>
    {
        private const string RENDER_TO_BITMAP_ITEM_NAME = "RenderToBitmap";
        private readonly IXpsRender _renderer;
        private readonly IFileProvider _fileProvider;

        private IFile _xpsDoc;
        
        [ImportingConstructor]
        public XpsRendererPlugin(IXpsRender renderer, IFileProvider fileProvider, IPilotDialogService dialogService)
        {
            var convertFromString = ColorConverter.ConvertFromString(dialogService.AccentColor);
            if (convertFromString != null)
            {
                var accentColor = (Color)convertFromString;
                ColorScheme.Initialize(accentColor, dialogService.Theme);
            }

            _renderer = renderer;
            _fileProvider = fileProvider;
        }

        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            var selection = context.SelectedObjects.ToList();
            if (context.IsContext || selection.Count != 1)
                return;

            var firstOrDefault = selection.FirstOrDefault();
            if (firstOrDefault != null)
                _xpsDoc = firstOrDefault.ActualFileSnapshot.Files.FirstOrDefault(f => IsXpsFile(f.Name));
            
            if (_xpsDoc != null)
                builder.AddItem(RENDER_TO_BITMAP_ITEM_NAME, 0).WithHeader("Plugin - RenderXpsToImage");
        }

        public void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            if (name != RENDER_TO_BITMAP_ITEM_NAME || !_fileProvider.Exists(_xpsDoc.Id))
                return;

            var xpsStream = _fileProvider.OpenRead(_xpsDoc);
            var model = new RenderedPagesViewModel(_renderer, xpsStream);
            var view = new RenderedPagesView
            {
                DataContext = model
            };
            view.ShowDialog();
            view.Closed += (sender, args) => model.Dispose();
        }

        private bool IsXpsFile(string fileName)
        {
            var extension = Path.GetExtension(fileName);
            return extension == ".xps";
        }
    }
}