/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Diagnostics;
using System.Linq;
using System.Windows.Controls;
using System.Windows.Data;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Controls;
using Ascon.Pilot.SDK.Controls.Commands;
using ChangeFilesMonitor.Tools;

namespace ChangeFilesMonitor
{
    class MainViewModel : PropertyChangedBase
    {
        private string _selectedItem;
        private readonly CollectionViewSource _editedFiles;
        private readonly DelegateCommand _sendToServerCommand;
        private readonly DelegateCommand _sendAllToServerCommand;
        private readonly DelegateCommand _navigateCommand;
        private readonly DelegateCommand _openCommand;
        private readonly IPilotStorageCommandController _pilotStorageCommandController;
        
        public MainViewModel(IPilotStorageCommandController pilotStorageCommandController)
        {
            _pilotStorageCommandController = pilotStorageCommandController;
            _editedFiles = new CollectionViewSource{ Source = EditedFilesListener.Current.EditedFiles };
            _editedFiles.SortDescriptions.Add(new SortDescription(string.Empty, ListSortDirection.Ascending));

            EditedFiles.CollectionChanged += OnEditedFilesChanged;

            _sendToServerCommand = new DelegateCommand(SendToServer, CanSend);
            _sendAllToServerCommand = new DelegateCommand(SendAllToServer, CanSendAll);
            _navigateCommand = new DelegateCommand(Navigate, CanNavigate);
            _openCommand = new DelegateCommand(Open, CanOpen);

            InitializeToolbar();
        }

        public ObservableCollection<ToolbarButtonItem> ToolbarItemsSource { get; } = new ObservableCollection<ToolbarButtonItem>();
        public ICollectionView EditedFiles => _editedFiles.View;

        public DelegateCommand SendAllToServerCommand => _sendAllToServerCommand;

        public string SelectedItem
        {
            get => _selectedItem;
            set
            {
                _selectedItem = value;
                NotifyOfPropertyChange(nameof(SelectedItem));
                _sendToServerCommand.RaiseCanExecuteChanged();
                _sendAllToServerCommand.RaiseCanExecuteChanged();
                _navigateCommand.RaiseCanExecuteChanged();
                _openCommand.RaiseCanExecuteChanged();
            }
        }

        private void OnEditedFilesChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            _sendAllToServerCommand.RaiseCanExecuteChanged();
        }

        private void SendAllToServer()
        {
            var editedFiles = EditedFilesListener.Current?.EditedFiles;
            if(editedFiles == null)
                return;

            _pilotStorageCommandController.Execute(PilotStorageCommand.Commit, editedFiles.ToArray());
        }

        private bool CanSendAll()
        {
            var editedFiles = EditedFilesListener.Current?.EditedFiles;
            if(editedFiles == null)
                return false;

            return editedFiles.Any() && _pilotStorageCommandController.CanExecute(PilotStorageCommand.Commit, editedFiles.ToArray());
        }

        private bool CanSend()
        {
            return !string.IsNullOrEmpty(SelectedItem) && _pilotStorageCommandController.CanExecute(PilotStorageCommand.Commit, SelectedItem);
        }

        private void SendToServer()
        {
            _pilotStorageCommandController.Execute(PilotStorageCommand.Commit, SelectedItem);
        }

        private void Navigate()
        {
            Process.Start("Explorer", @"/e,/select,""" + SelectedItem + @"""");
        }

        private bool CanNavigate()
        {
            return !string.IsNullOrEmpty(SelectedItem);
        }

        private void Open()
        {
            Process.Start(SelectedItem);
        }

        private bool CanOpen()
        {
            return !string.IsNullOrEmpty(SelectedItem);
        }


        private void InitializeToolbar()
        {
            ToolbarItemsSource.Add(new ToolbarButtonItem("SendAll")
            {
                Header = "Отправить все",
                Icon = Icons.Instance.SendAllIcon,
                Command = _sendAllToServerCommand
            });

            ToolbarItemsSource.Add(new ToolbarButtonItem("SendSelected")
            {
                Header = "Отправить выделенный",
                Icon = Icons.Instance.SendSelectedIcon,
                Command = _sendToServerCommand
            });

            ToolbarItemsSource.Add(new ToolbarButtonItem("Navigate")
            {
                Header = "Показать в проводнике",
                Icon = Icons.Instance.ShowFilesOnDiskIcon,
                Command = _navigateCommand
            });
        }

        public void BuildContextMenu(ContextMenu menu)
        {
            menu.Items.Add(new MenuItem
            {
                Header = "Открыть",
                Command = _openCommand
            });
            menu.Items.Add(new MenuItem
            {
                Header = "Показать в проводнике",
                Icon = Icons.Instance.ShowFilesOnDiskIcon,
                Command = _navigateCommand
            });
            menu.Items.Add(new MenuItem
            {
                Header = "Отправить",
                Icon = Icons.Instance.SendSelectedIcon,
                Command = _sendToServerCommand
            });
        }
    }
}
