/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls.Primitives;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Menu;
using Ascon.Pilot.SDK.ObjectCard;
using Ascon.Pilot.Theme.Controls.SearchBox.Grammar;

namespace CustomXpsAnnotations
{
    [Export(typeof(IXpsAnnotationHandler))]
    [Export(typeof(IMenu<XpsRenderClickPointContext>))]
    public class XpsAdornerHandler : IXpsAnnotationHandler, IMenu<XpsRenderClickPointContext>
    {
        private readonly IXpsViewer _viewer;
        private readonly IObjectsRepository _repository;
        private readonly ISearchService _searchService;
        private readonly IRemarkModifier _modifier;
        private readonly IObjectCardControlProvider _objectCardControlProvider;
        private IType _customAnnotationType;
        private const string CUSTOM_XPS_ANNOTATION_COMMAND_NAME = "CustomXpsAnnotationCommandName";
        private const string CUSTOM_REMARK_TYPE_NAME = "CustomXpsAnnotation";

        [ImportingConstructor]
        public XpsAdornerHandler(IXpsViewer viewer, IObjectsRepository repository, ISearchService searchService, IRemarkModifier modifier,
            IPilotDialogService dialogService, IAttributeFormatParser formatParser, IObjectCardControlProvider objectCardControlProvider)
        {
            _customAnnotationType = repository.GetType(CUSTOM_REMARK_TYPE_NAME);
            _viewer = viewer;
            _repository = repository;
            _searchService = searchService;
            _modifier = modifier;
            _objectCardControlProvider = objectCardControlProvider;
        }

        public XpsAnnotationAdorner GetAnnotationAdorner(DocumentPageView adornedElement, XpsAnnotation annotation, bool isMine)
        {
            if (_customAnnotationType == null)
                return null;
            if(_customAnnotationType.Id == -1)
                _customAnnotationType = _repository.GetType(CUSTOM_REMARK_TYPE_NAME);
            var builder = _searchService.GetAnnotationQueryBuilder().WithCurrentVersion().WithType(_customAnnotationType.Id);
            _viewer.SetAnnotationsFilter("filter example", builder.ToString());

            return new CustomXpsAnnotationAdorner(adornedElement, annotation, _modifier, _repository, _objectCardControlProvider);
        }

        public void Build(IMenuBuilder builder, XpsRenderClickPointContext context)
        {
            var person = _repository.GetCurrentPerson();
            var accessLevel = _repository.GetCurrentAccess(context.DataObject.Id, person.Id);

            builder.AddItem(CUSTOM_XPS_ANNOTATION_COMMAND_NAME, 6)
                .WithHeader("Add custom xps annotation")
                .WithIsEnabled(accessLevel.HasFlag(AccessLevel.Agreement));
        }

        public async void OnMenuItemClick(string name, XpsRenderClickPointContext context)
        {
            Console.WriteLine(context.ClickPoint);
            if (name.Equals(CUSTOM_XPS_ANNOTATION_COMMAND_NAME))
            {
                var databaseId = _repository.GetDatabaseId();
                var person = _repository.GetCurrentPerson();
                var annotation = new XpsAnnotation
                {
                    AnnotationId = Guid.NewGuid(),
                    PositionX = context.ClickPoint.X,
                    PositionY = context.ClickPoint.Y,
                    Mark = XpsAnnotationsHelper.GetMark(person.ActualName),
                    PageNumber = context.PageNumber,
                    TypeId = _customAnnotationType.Id,
                    Kind = "BlueAnnotation",
                };

                await _modifier.CreateRemark(context.DataObject, SystemTypeNames.DOCUMENT_REMARKS_FOLDER, CUSTOM_REMARK_TYPE_NAME, annotation);
                _modifier.Apply();
            }
        }
    }
}
