/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.Composition;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Media;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.ObjectCard;
using Ascon.Pilot.Theme.ColorScheme;
using Ascon.Pilot.Theme.Controls;
using Ascon.Pilot.Themes;
using ProjectCloneWizard.ObjectsTree;
using ProjectCloneWizard.ViewModels;
using IDataObject = Ascon.Pilot.SDK.IDataObject;

namespace ProjectCloneWizard.Common
{
    public interface IProjectCloneCreator
    {
        void ShowWizardDialog(IDataObject selected);
        string Icon { get; }
    }

    [Export(typeof(IProjectCloneCreator))]
    class ProjectCloneCreator : IProjectCloneCreator
    {
        private readonly IObjectsRepository _repository;
        private readonly IPilotDialogService _pilotDialogService;
        private readonly IAttributeFormatParser _attributeFormatParser;
        private readonly IFileProvider _fileProvider;
        private readonly ITabServiceProvider _tabServiceProvider;
        private readonly IObjectModifier _modifier;
        private CreationInfo _creationInfo;
        private readonly List<IDisposable> _disposables = new List<IDisposable>();
        private readonly IObjectCardControlProvider _cardControlProvider;

        [ImportingConstructor]
        public ProjectCloneCreator(IObjectsRepository repository, IPilotDialogService pilotDialogService, IAttributeFormatParser attributeFormatParser, 
            IObjectModifier modifier, IFileProvider fileProvider, ITabServiceProvider tabServiceProvider, IObjectCardControlProvider cardControlProvider)
        {
            _cardControlProvider = cardControlProvider;
            _repository = repository;
            _pilotDialogService = pilotDialogService;
            _attributeFormatParser = attributeFormatParser;
            _fileProvider = fileProvider;
            _tabServiceProvider = tabServiceProvider;
            _modifier = modifier;

            InitColorScheme(pilotDialogService);

            Icon = "menu_icon_dark.svg";
            if (pilotDialogService.Theme == ThemeNames.Jedi)
                Icon = "menu_icon_light.svg";
        }

        public void ShowWizardDialog(IDataObject selected)
        {
            if (selected == null)
                return;

            var title = "Мастер создания элемента по шаблону";
            if (selected.Type.IsProject)
                title = "Мастер создания структуры по шаблону";
            
            //Создадим контейнер для сбора информации о выбранных параметрах
            _creationInfo = new CreationInfo
            {
                Parent = selected,
            };

            var view = new WizardControl();
            FillWizardPages(view);

            var dialog = new PureWindow
            {
                Content = view,
                Title = title,
                WindowStartupLocation = WindowStartupLocation.CenterScreen,
                Width = 800,
                Height = 600,
                ShowInTaskbar = true,
                Uid = "ProjectCloneWizardWindow"
            };

            dialog.Closing += DialogOnClosing;
            dialog.Show();
        }

        private void DialogOnClosing(object sender, CancelEventArgs cancelEventArgs)
        {
            var result = (bool)((PureWindow) sender).GetValue(Ascon.Pilot.Theme.Controls.Windows.Tools.WindowExtensions.CloseNonModalWindowResultProperty);
            if (_creationInfo.ObjectIdToShow != default(Guid) && _creationInfo.Exception == null && result)
            {
                _tabServiceProvider.ShowElement(_creationInfo.ObjectIdToShow);
            }

            foreach (var disposable in _disposables)
            {
                disposable?.Dispose();
            }
        }

        public string Icon { get; }

        private void FillWizardPages(WizardControl view)
        {
            //Наполним визард страницами
            var isUpdateStruct = _creationInfo.Parent.Type.IsProject;

            //Страница выбора шаблона
            var filter = new ListViewFolderChildrenFilter(_repository);
            var dataObjectService = new DataObjectService(_repository, filter);
            _disposables.Add(dataObjectService);
            var vm = new SelectProjectTemplatePageViewModel(_creationInfo, dataObjectService);
            _disposables.Add(vm);
            view.Pages.Add(new SelectProjectTemplatePage { DataContext = vm });

            //Страница состав выбранного проекта
            var serviceFactory = new ListViewDataObjectServiceFactory(_repository);
            var vm2 = new ProjectCompositionPageViewModel(_creationInfo, serviceFactory);
            _disposables.Add(vm2);
            view.Pages.Add(new ProjectCompositionPage { DataContext = vm2 });

            //Страница файлы и папки выбранного проекта
            var storageServiceFactory = new StorageDataObjectServiceFactory(_repository);
            var vm3 = new FilesAndFolderPageViewModel(_creationInfo, storageServiceFactory);
            _disposables.Add(vm3);
            view.Pages.Add(new FilesAndFolderPage { DataContext = vm3 });

            //Страница карточки нового проекта
            if (!isUpdateStruct)
            {
                var vm4 = new ProjectCardPageViewModel(_creationInfo, _cardControlProvider);
                view.Pages.Add(new ProjectCardPage {DataContext = vm4});
            }

            //Страница процесс
            var processViewModel = isUpdateStruct
                ? new UpdateStructProcessPageViewModel(_creationInfo, _modifier, _repository, _fileProvider) 
                : new ProcessPageViewModel(_creationInfo, _modifier, _repository, _fileProvider);
            
            view.Pages.Add(new ProcessPage { DataContext = processViewModel });

            //Результат
            var vm5 = new FinishPageViewModel(_creationInfo);
            view.Pages.Add(new FinishPage { DataContext = vm5 });
        }

        private void InitColorScheme(IPilotDialogService pilotDialogService)
        {
            var accentColorObj = ColorConverter.ConvertFromString(pilotDialogService.AccentColor);
            if (accentColorObj == null)
                return;

            var accentColor = (Color)accentColorObj;
            ColorScheme.Initialize(accentColor, pilotDialogService.Theme);
        }
    }
}
