/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace PilotShare.Client.Utils
{
    public static class Constants
    {
        public const string PROJECT_TITLE_ATTRIBUTES_DELIMITER = " - ";
        public const int MAX_PROJECT_ITEM_FOLDER_NAME_LENGTH = 40;
        public const string TRUNCATED_NAME_MARK = "~";
        public const string ANNOTATIONS_DEFINITION = "Annotation";
        public const string GRAPHIC_LAYER_ELEMENT_DEFINITION = "PILOT_GRAPHIC_LAYER_ELEMENT_";
        public const string GRAPHIC_LAYER_ELEMENT_CONTENT = "PILOT_CONTENT_GRAPHIC_LAYER_ELEMENT_";
        public const string ANNOTATION_CHAT_MESSAGE = "Note_Chat_Message";
        public const string DIGITAL_SIGNATURE = "PilotDigitalSignature";
        public const string THUMBNAIL_FILE_NAME_POSTFIX = "PilotThumbnail";
        public const string TEXT_LABELS_DEFINITION = "PilotTextLabels";
        public const string BARCODE_DEFINITION = "PilotBarcode";
        public static readonly char[] Separators = { ' ', '.', ',', '"', '\'', '!', '?', ':', ';' };
        public const int MAX_ITEMS_LOAD_PER_PAGE = 250;
        public const int MAX_TASKS_LOAD_PER_PAGE = 1000;
        public const int NEXT_TASKS_LOAD_PER_PAGE = 300;
        public const string TASK_FILE_ATTACHMENTS_PATH_SERIALIZATION_ID = "TaskFileAttachmentsDialogPath";
        public const string TASK_DIRECTORY_ATTACHMENTS_PATH_SERIALIZATION_ID = "TaskDirectoryAttachmentsDialogPath";
        public const string LINKED_OBJECTS_ATTACHMENTS_PATH_SERIALIZATION_ID = "LinkedObjectsAttachmentsDialogPath";
        public const string LINKED_TO_CHAT_OBJECTS_PATH_SERIALIZATION_ID = "LinkedToChatDialogPath";
        public const string SOURCE_FILE_PATH_SERIALIZATION_ID = "SourceFileDialogPath";
        public const int CHATS_PAGE_COUNT = 30;
        public const int MESSAGES_PAGE_COUNT = 30;
        public const int CHAT_MEMBERS_PAGE_COUNT = 30;
        public const int MAX_TEXT_MESSAGE_LENGTH = 2500;
        public const int SERVER_ADMINISTRATOR_PERSON_ID = -1;
        public const int SERVER_ACTION_PERSON_ID = -2;
    }

    public static class FileExtensionHelper
    {
        public const string XPS_EXTENSION = ".xps";
        private const string PDF_ALIKE_REGEX_MASK = @"\.pdf$";
        private const string SUPPORTED_DOCUMENT_REGEX_MASK = @"\.xps$|\.pdf$|\.dwfx$";
        public const string XPS_ALIKE_REGEX_MASK = @"\.xps$|\.dwfx$";
        private const string GRAPHIC_FILE_REGEX_MASK = @"\.jpg$|\.jpeg$|\.png$|\.tiff$|\.tif$|\.bmp$";
        private const string EMAIL_FILE_REGEX_MASK = @"\.eml$|\.msg$";
        private const string MSG_FILE_REGEX_MASK = @"\.msg$";
        public const string BATCH_PRINT_LIST_EXTENSION = ".plst";
        public const string BATCH_PRINT_LIST_EXTENSION_FILTER = "(*.plst)|*.plst";

        public static bool IsSupportedDocument(string filePath)
        {
            return IsFileMatchesExtensionMask(filePath, SUPPORTED_DOCUMENT_REGEX_MASK);
        }

        public static List<string> GetElementsFromMask(string regexString)
        {
            if (String.IsNullOrEmpty(regexString))
                return new List<string>();

            var elementsMatches = Regex.Matches(regexString, @"\..*?(?=\$)");
            var elements = elementsMatches.Cast<Match>().Select(match => match.Value).ToList();
            return elements;
        }

        public static List<string> GetSupportedDocumentExtensions()
        {
            return GetElementsFromMask(SUPPORTED_DOCUMENT_REGEX_MASK);
        }

        public static bool IsSupportedDocumentForSave(string filePath)
        {
            return !string.IsNullOrEmpty(filePath) &&
                   !IsSystemFile(filePath) &&
                   !IsBarcodeFile(filePath) &&
                   !IsTextLabelsFile(filePath);
        }

        public static bool IsPdfAlike(string filePath)
        {
            return IsFileMatchesExtensionMask(filePath, PDF_ALIKE_REGEX_MASK);
        }

        public static bool IsXpsAlike(string filePath)
        {
            return IsFileMatchesExtensionMask(filePath, XPS_ALIKE_REGEX_MASK);
        }

        public static bool IsGraphicFile(string filePath)
        {
            return IsFileMatchesExtensionMask(filePath, GRAPHIC_FILE_REGEX_MASK);
        }

        public static bool IsEmailFile(string filePath)
        {
            return IsFileMatchesExtensionMask(filePath, EMAIL_FILE_REGEX_MASK);
        }

        public static bool IsMsgEmailFile(string filePath)
        {
            return IsFileMatchesExtensionMask(filePath, MSG_FILE_REGEX_MASK);
        }

        public static bool IsFileMatchesExtensionMask(string filePath, string mask)
        {
            var extension = Path.GetExtension(filePath);
            return !string.IsNullOrEmpty(extension) && Regex.IsMatch(extension, mask, RegexOptions.IgnoreCase);
        }

        public static bool IsAnnotationFile(string fileName)
        {
            return fileName.Contains(Constants.ANNOTATIONS_DEFINITION);
        }

        public static bool IsGraphicLayerElementFile(string fileName)
        {
            return fileName.Contains(Constants.GRAPHIC_LAYER_ELEMENT_DEFINITION);
        }

        public static bool IsGraphicLayerElementContentFile(string fileName)
        {
            return fileName.Contains(Constants.GRAPHIC_LAYER_ELEMENT_CONTENT);
        }

        public static bool IsAnnotationMessageFile(string fileName)
        {
            return fileName.Contains(Constants.ANNOTATION_CHAT_MESSAGE);
        }

        public static bool IsDigitalSignatureFile(string fileName)
        {
            return fileName.Equals(Constants.DIGITAL_SIGNATURE);
        }

        public static bool IsBarcodeFile(string fileName)
        {
            return fileName.Equals(Constants.BARCODE_DEFINITION);
        }

        public static bool IsTextLabelsFile(string fileName)
        {
            return fileName.Equals(Constants.TEXT_LABELS_DEFINITION);
        }

        public static bool IsInjectingFile(string filename)
        {
            return IsTextLabelsFile(filename) || IsBarcodeFile(filename);
        }

        public static bool IsThumbnailAlike(string filename)
        {
            return filename.EndsWith(Constants.THUMBNAIL_FILE_NAME_POSTFIX);
        }

        public static bool IsSystemFile(string filename)
        {
            return IsAnnotationFile(filename)
                   || IsDigitalSignatureFile(filename)
                   || IsThumbnailAlike(filename);
        }
    }
}
